/*
 * This file is part of TelepathyQt4
 *
 * Copyright (C) 2010 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "tube-channel.h"
#include <TelepathyQt4/PendingOperation>

namespace Tp {

/** 
 * \class PendingAcceptStreamTube
 * \headerfile TelepathyQt4/stream-tube.h <TelepathyQt4/StreamTube>
 *
 * \brief A pending operation for accepting a stream tube
 *
 * This class represents an asynchronous operation for accepting a stream tube.
 * When the operation is finished, you can access the resulting socket address
 * through #address().
 */
class TELEPATHY_QT4_EXPORT PendingAcceptStreamTube : public PendingOperation
{
    Q_OBJECT
    Q_DISABLE_COPY(PendingAcceptStreamTube)

public:
    PendingAcceptStreamTube(QObject *parent = 0);
    ~PendingAcceptStreamTube();

    /**
     * If the operation was successful, returns the address the connection manager
     * should listen to. Please note that until the tube reaches the \c Open state,
     * you should not attempt to connect to the socket.
     */
    QVariant address() const;

private:
    friend class StreamTube;

    struct Private;
    friend struct Private;
    Private *mPriv;
};

/**
 * \class StreamTube
 * \headerfile TelepathyQt4/stream-tube.h <TelepathyQt4/StreamTube>
 *
 * \brief A class representing a Stream Tube
 *
 * \c StreamTube is an high level wrapper for managing Telepathy interface
 * org.freedesktop.Telepathy.Channel.Type.StreamTube.
 * It provides a transport for reliable and ordered data transfer, similar to SOCK_STREAM sockets.
 *
 * For more details, please refer to Telepathy spec.
 */
class StreamTube : public Channel
{
    Q_OBJECT
    Q_DISABLE_COPY(StreamTube)

public:
    /**
     * Feature used in order to monitor connections to this tube.
     *
     * newRemoteConnection will be emitted when a new remote connection gets opened
     * newLocalConnection will be emitted when a new local connection gets opened
     * connectionClosed will be emitted when an existing connection gets closed
     */
    static const Feature FeatureMonitorConnections;

    /**
     * Create a new StreamTube channel.
     *
     * \param connection Connection owning this channel, and specifying the
     *                   service.
     * \param objectPath The object path of this channel.
     * \param immutableProperties The immutable properties of this channel.
     * \return A StreamTubePtr object pointing to the newly created
     *         StreamTube object.
     */
    static StreamTubePtr create(const ConnectionPtr &connection,
            const QString &objectPath, const QVariantMap &immutableProperties);

    /**
     * \returns the service name that will be used over the tube
     */
    QString service() const;

    /**
     * \returns A mapping from address types to lists of access-control type
     *          that the connection manager supports for stream tubes with that address type
     */
    SupportedSocketMap supportedSocketTypes() const;

public Q_SLOTS:
    /**
     * Offer a stream tube exporting the local socket specified
     *
     * \param socketType The type of the listening address of the local service
     * \param address The listening address of the local service, as indicated by socketType
     * \param accessControl The access control the local service applies to the local socket
     * \param parameters The dictionary of arbitrary Parameters to send with the tube offer.
     *
     * \returns a pending operation for monitoring the request
     */
    PendingOperation *offer(SocketAddressType socketType, const QVariant &address,
                            SocketAccessControl accessControl, const QVariantMap &parameters);

    /**
     * Accept a stream tube. This function has effect only upon tubes in \c LocalPending state.
     * When this function is called, the connection manager will attempt opening the tube,
     * and will return an address for listening connections to upon success.
     *
     * \note After this operation has been completed successfully, you must \b not try to connect
     *       to the returned address until the Tube state changes to \c Open.
     *
     * \param socketType The type of address the connection manager should listen on.
     * \param accessControl The type of access control the connection manager should apply to the socket.
     * \param controlParameter A parameter for the access control type, depending on the specified
     *                         accessControl type.
     *
     * \returns a \c PendingAcceptStreamTube for monitoring the request and obtaining the socket address
     *          upon success.
     */
    PendingAcceptStreamTube *accept(SocketAddressType socketType, SocketAccessControl accessControl,
                                    const QVariant &controlParameter);

protected:
    StreamTube(const ConnectionPtr &connection, const QString &objectPath,
            const QVariantMap &immutableProperties);

Q_SIGNALS:
    void newRemoteConnection(uint contactHandle, const QVariant &parameter, uint connectionId);
    void newLocalConnection(uint connectionId);
    void connectionClosed(uint connectionId, const QString &error, const QString &message);
};

}