/*
 * This file is part of TelepathyQt4
 *
 * Copyright (C) 2010 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "tube-channel.h"
#include <TelepathyQt4/PendingOperation>

namespace Tp {

/**
 * \class PendingAcceptDBusTube
 * \headerfile TelepathyQt4/dbus-tube.h <TelepathyQt4/DBusTube>
 *
 * \brief A pending operation for accepting a dbus tube
 *
 * This class represents an asynchronous operation for accepting a dbus tube.
 * When the operation is finished, you can access the resulting private bus address
 * through #address().
 */
class TELEPATHY_QT4_EXPORT PendingAcceptDBusTube : public PendingOperation
{
    Q_OBJECT
    Q_DISABLE_COPY(PendingAcceptDBusTube)

public:
    PendingAcceptDBusTube(QObject *parent = 0);
    ~PendingAcceptDBusTube();

    /**
     * If the operation was successful, returns the obtained private bus address.
     * Please note that until the tube reaches the \c Open state,
     * you should not attempt to connect to the returned bus.
     */
    QString address() const;

private:
    friend class DBusTube;

    struct Private;
    friend struct Private;
    Private *mPriv;
};

/**
 * \class PendingOfferDBusTube
 * \headerfile TelepathyQt4/dbus-tube.h <TelepathyQt4/DBusTube>
 *
 * \brief A pending operation for offering a dbus tube
 *
 * This class represents an asynchronous operation for offering a dbus tube.
 * When the operation is finished, you can access the resulting private bus address
 * through #address().
 */
class TELEPATHY_QT4_EXPORT PendingOfferDBusTube : public PendingOperation
{
    Q_OBJECT
    Q_DISABLE_COPY(PendingOfferDBusTube)

public:
    PendingOfferDBusTube(QObject *parent = 0);
    ~PendingOfferDBusTube();

    /**
     * If the operation was successful, returns the obtained private bus address.
     * Please note that until the tube reaches the \c Open state,
     * you should not attempt to connect to the returned bus.
     */
    QString address() const;

private:
    friend class DBusTube;

    struct Private;
    friend struct Private;
    Private *mPriv;
};

/**
 * \class DBusTube
 * \headerfile TelepathyQt4/dbus-tube.h <TelepathyQt4/DBusTube>
 *
 * \brief A class representing a DBus Tube
 *
 * \c DBusTube is an high level wrapper for managing Telepathy interface
 * org.freedesktop.Telepathy.Channel.Type.DBusTube.
 * It provides an ordered, reliable transport for DBus traffic.
 *
 * For more details, please refer to Telepathy spec.
 */
class DBusTube : public Channel
{
    Q_OBJECT
    Q_DISABLE_COPY(DBusTube)

public:
    /**
     * Feature used in order to monitor bus names changes in this tube.
     *
     * busNamesChanged will be emitted when known service names in the bus changes.
     */
    static const Feature FeatureMonitorBusNames;

    /**
     * Create a new DBusTube channel.
     *
     * \param connection Connection owning this channel, and specifying the
     *                   service.
     * \param objectPath The object path of this channel.
     * \param immutableProperties The immutable properties of this channel.
     * \return A DBusTubePtr object pointing to the newly created
     *         DBusTube object.
     */
    static DBusTubePtr create(const ConnectionPtr &connection,
            const QString &objectPath, const QVariantMap &immutableProperties);

    /**
     * \returns A string representing the service name that will be used over the tube.
     *
     * \note This string always represents a valid and well-known DBus service name.
     */
    QString serviceName() const;

    /**
     * \returns A mapping between contact handles and their unique bus names on this tube.
     *
     * \note This method will return an empty dictionary if this tube is operating in
     *       peer-to-peer mode.
     */
    DBusTubeParticipants busNames() const;

    /**
     * \returns A list of the access control types that are supported with this channel.
     */
    QList< SocketAccessControl > supportedAccessControls() const;

    /**
     * NOTE FOR REVIEWERS: I don't know if implementing this method is feasible and/or useful
     *
     * Returns a QDBusConnection representing the DBus connection opened by this tube
     */
    QDBusConnection connection() const;

public Q_SLOTS:
    /**
     * Offer a DBus tube, providing the specified service
     *
     * \note After this operation has been completed successfully, you must \b not try to connect
     *       to the returned address until the Tube state changes to \c Open.
     *
     * \param parameters The dictionary of arbitrary Parameters to send with the tube offer.
     * \param accessControl The access control the connection manager applies to the D-Bus socket.
     *
     * \returns a \c PendingOfferDBusTube for monitoring the request and obtaining the private
     *          bus address upon success.
     */
    PendingOfferDBusTube *offer(const QVariantMap &parameters, SocketAccessControl accessControl);

    /**
     * Accept a DBus tube. This function has effect only upon tubes in \c LocalPending state.
     * When this function is called, the connection manager will attempt opening the tube,
     * and will return a private bus address upon success.
     *
     * \note After this operation has been completed successfully, you must \b not try to connect
     *       to the returned address until the Tube state changes to \c Open.
     *
     * \param accessControl The access control the connection manager applies to the D-Bus socket.
     *
     * \returns a \c PendingAcceptDBusTube for monitoring the request and obtaining the private
     *          bus address upon success.
     */
    PendingAcceptDBusTube *accept(SocketAccessControl accessControl);

protected:
    DBusTube(const ConnectionPtr &connection, const QString &objectPath,
            const QVariantMap &immutableProperties);

Q_SIGNALS:
    void busNamesChanged(const DBusTubeParticipants &added, const UIntList &removed);
};

}
