/*
 * Copyright (C) 2006 Wolfgang Grandegger <wg@grandegger.com>
 *
 * Copyright (C) 2005, 2006 Sebastian Smolorz
 *                          <Sebastian.Smolorz@stud.uni-hannover.de>
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; eitherer version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <linux/module.h>
#include <linux/ioport.h>
#include <linux/delay.h>

#include <rtdm/rtdm_driver.h>

#include <rtdm/rtcan.h>
#include <rtcan_dev.h>
#include <rtcan_raw.h>
#include <rtcan_internal.h>
#include <rtcan_sja1000.h>
#include <rtcan_sja1000_regs.h>

#define RTCAN_DEV_NAME    "rtcan%d"
#define RTCAN_DRV_NAME    "sja1000-tscan1"

#define TSCAN1_BASE_ADDR  0x150

#define RTCAN_TSCAN1_MAX_DEV 4
#define RTCAN_IRQ5        0x30
#define RTCAN_IRQ6        0x10
#define RTCAN_IRQ7        0x20

static char *tscan1_board_name = "TSCAN1-Board";

MODULE_AUTHOR("Modified from rtcan_isa.c for TS_CAN1 John Charlton <john.charlton@tycoelectronics.com>");
MODULE_DESCRIPTION("RTCAN board driver for standard TSCAN1 boards");
MODULE_SUPPORTED_DEVICE("TSCAN1 board");
MODULE_LICENSE("GPL");

static u16 io[RTCAN_TSCAN1_MAX_DEV];
static int irq[RTCAN_TSCAN1_MAX_DEV];
static u32 can_clock[RTCAN_TSCAN1_MAX_DEV];
static u8 ocr[RTCAN_TSCAN1_MAX_DEV];
static u8 cdr[RTCAN_TSCAN1_MAX_DEV];

compat_module_param_array(io, ushort, RTCAN_TSCAN1_MAX_DEV, 0444);
compat_module_param_array(irq, int, RTCAN_TSCAN1_MAX_DEV, 0444);
compat_module_param_array(can_clock, uint, RTCAN_TSCAN1_MAX_DEV, 0444);
compat_module_param_array(ocr, byte, RTCAN_TSCAN1_MAX_DEV, 0444);
compat_module_param_array(cdr, byte, RTCAN_TSCAN1_MAX_DEV, 0444);

MODULE_PARM_DESC(io, "The io-port address");
MODULE_PARM_DESC(irq, "The interrupt number");
MODULE_PARM_DESC(can_clock, "External clock frequency (default 16 MHz)");
MODULE_PARM_DESC(ocr, "Value of output control register (default 0x1a)");
MODULE_PARM_DESC(cdr, "Value of clock divider register (default 0xc8");

#define RTCAN_TSCAN1_PORT_SIZE 32

struct rtcan_tscan1
{
  u16 io;
};

static struct rtcan_device *rtcan_tscan1_devs[RTCAN_TSCAN1_MAX_DEV];

static u8 rtcan_tscan1_readreg(struct rtcan_device *dev, int port)
{
  u8 ret = 0;
  struct rtcan_tscan1 *board = (struct rtcan_tscan1 *)dev->board_priv;
  ret = inb(board->io + port);
  
  return ret;
}

static void rtcan_tscan1_writereg(struct rtcan_device *dev, int port, u8 val)
{
  struct rtcan_tscan1 *board = (struct rtcan_tscan1 *)dev->board_priv;
  outb(val, board->io + port);
}


int __init rtcan_tscan1_init_one(int idx)
{
  struct rtcan_device *dev;
  struct rtcan_sja1000 *chip;
  struct rtcan_tscan1 *board;
  int ret;
  u8 jumper_set = 0;

  if ((dev = rtcan_dev_alloc(sizeof(struct rtcan_sja1000),
                             sizeof(struct rtcan_tscan1))) == NULL)
    return -ENOMEM;

  chip = (struct rtcan_sja1000 *)dev->priv;
  board = (struct rtcan_tscan1 *)dev->board_priv;

  dev->board_name = tscan1_board_name;

  printk("Initializing board %2d\n", idx);
  
  /* Check and request I/O ports for SJA1000 I/O range selection */
  if (!request_region(TSCAN1_BASE_ADDR + idx * 8, 8, RTCAN_DRV_NAME)) {
    ret = -EBUSY;
    goto out_dev_free;
  }

  outb(0x0, TSCAN1_BASE_ADDR + idx * 8 + 4);
  outb(0x60 + idx, TSCAN1_BASE_ADDR + idx * 8 + 5); // set base I/O address in
                                                    // control reg
  jumper_set = inb(TSCAN1_BASE_ADDR + idx * 8 + 6);

  board->io = io[idx];

  chip->irq_num = irq[idx];
  chip->irq_flags = 0;  //RTDM_IRQTYPE_EDGE; //RTDM_IRQTYPE_SHARED | 

  chip->read_reg = rtcan_tscan1_readreg;
  chip->write_reg = rtcan_tscan1_writereg;

  /* Check and request I/O ports */
  if (!request_region(board->io, RTCAN_TSCAN1_PORT_SIZE, RTCAN_DRV_NAME)) {
    ret = -EBUSY;
    goto out_baseio_free;
  }

  /* Clock frequency in Hz */
  if (can_clock[idx])
    dev->can_sys_clock = can_clock[idx] / 2;
  else
    dev->can_sys_clock = 8000000; /* 16/2 MHz */

  
  /* Output control register */
  if (ocr[idx])
    chip->ocr = ocr[idx];
  else
    chip->ocr = SJA_OCR_MODE_NORMAL | SJA_OCR_TX0_PUSHPULL;

  if (cdr[idx])
    chip->cdr = cdr[idx];
  else
    chip->cdr = SJA_CDR_CAN_MODE | SJA_CDR_CLK_OFF | SJA_CDR_CBP;

  strncpy(dev->name, RTCAN_DEV_NAME, IFNAMSIZ);

  ret = rtcan_sja1000_register(dev);
  if (ret) {
    printk(KERN_ERR "ERROR %d while trying to register SJA1000 "
           "device!\n", ret);
    goto out_free_region;
  }

  rtcan_tscan1_devs[idx] = dev;
  return 0;

out_free_region:
  release_region(board->io, RTCAN_TSCAN1_PORT_SIZE);

out_baseio_free:
  release_region(TSCAN1_BASE_ADDR, 8);

out_dev_free:
  rtcan_dev_free(dev);

  return ret;
}

static void rtcan_tscan1_exit(void);

/** Init module */
static int __init rtcan_tscan1_init(void)
{
  int i, err;
  int devices = 0;

  for (i = 0; i < RTCAN_TSCAN1_MAX_DEV && io[i] != 0; i++) {
    err = rtcan_tscan1_init_one(i);
    if (err) {
      rtcan_tscan1_exit();
      return err;
    }
    devices++;
  }
  if (devices)
    return 0;

  printk(KERN_ERR "ERROR! No devices specified! "
         "Use io=<port1>[,...] irq=<irq1>[,...]\n");
  return -EINVAL;
}


/** Cleanup module */
static void rtcan_tscan1_exit(void)
{
  int i;
  struct rtcan_device *dev;
  struct rtcan_tscan1 *board;
  
  for (i = 0; i < RTCAN_TSCAN1_MAX_DEV; i++) {
    printk("Releasing board %2d\n", i);
  
    dev = rtcan_tscan1_devs[i];
    if (!dev)
      continue;
    rtcan_sja1000_unregister(dev);
    board = (struct rtcan_tscan1 *)dev->board_priv;
    if (board) {
      release_region(board->io, RTCAN_TSCAN1_PORT_SIZE);
    }
    rtcan_dev_free(dev);
    release_region(TSCAN1_BASE_ADDR + i * 8, 8);
  }
}

module_init(rtcan_tscan1_init);
module_exit(rtcan_tscan1_exit);
