/*
 * Silverscreen
 * (C) 2010 William Lahti.
 *
 */

using System;
using System.IO;
using System.Threading;
using System.Runtime.InteropServices;
using System.Reflection;
using System.Globalization;
using System.Collections.Generic;

using Tao.OpenGl;
using Tao.DevIl;
using GtkGL;

namespace Silverscreen {
	public class TextureManager {
		public TextureManager ()
		{
			this.TexturePath = new List<string> ();
		}

		List<Texture> textures = new List<Texture> ();
		List<Texture> reloadSchedule = new List<Texture> ();
		
		protected bool texturesLoaded = false;
		public Texture BlankTexture { get; protected set; }
		protected List<TextureInfo> texturesToLoad = new List<TextureInfo> ();
				
		protected class TextureInfo {
			public TextureInfo (string file, string member)
			{
				File = file;
				Member = member;
			}

			public string File { get; set; }
			public string Member { get; set; }
		}

		protected virtual void ReportLoadProgress (string file, float progress)
		{
		}
		
		public virtual void LoadTextures ()
		{
			if (texturesLoaded)
				return;
			texturesLoaded = true;
						
			int x = 0, max = texturesToLoad.Count;
			foreach (var tex in texturesToLoad) {
				ReportLoadProgress (tex.File, x / (float)max);
				var prop = GetType().GetProperty (tex.Member);
				prop.SetValue (this, Load (tex.File), new object[0]);
				++x;
			}
		}

		public List<string> TexturePath { get; set; }
		Dictionary<string,Texture> namedTextures { get; set; }

		public Texture this [string name] {
			get { return namedTextures[name]; }
			set { namedTextures[name] = value; }
		}
		
		public string FindTexture (string name)
		{
			if (File.Exists (name))
				return name;
			
			foreach (string path in TexturePath) {
				if (File.Exists (Path.Combine (path, name)))
					return Path.Combine(path, name);
			}

			return name;
		}
		
		public Texture Load (string filename)
		{
			var t = new Texture (this, FindTexture(filename));
			lock (textures)
				textures.Add (t);
			return t;
		}

		public void ManageTexture (Texture tex)
		{
			lock (textures)
				textures.Add (tex);
		}

		public void AbandonTexture (Texture tex)
		{
			lock (textures)
				textures.Remove (tex);
		}
		
		/// <summary>
		/// Schedule the given texture to be reloaded upon the next
		/// call to RunIteration(). This should only be called when
		/// reloading a texture directly is not possible, such as
		/// while rendering the frame or running in a different thread
		/// than the GL renderer.
		/// </summary>
		public void ScheduleReload (Texture tex)
		{
			lock (textures) {
				if (!textures.Contains(tex))
					textures.Add(tex);
			}

			lock (reloadSchedule) {
				reloadSchedule.Add (tex);
			}
		}

		public void RunIteration ()
		{
			Texture toReload = null;
			
			lock (reloadSchedule) {
				if (reloadSchedule.Count > 0) {
					toReload = reloadSchedule[reloadSchedule.Count-1];
					reloadSchedule.RemoveAt(reloadSchedule.Count-1);
				}
			}

			if (toReload != null)
				toReload.Reload();
		}
		
		public void ReloadAll()
		{
			foreach (var t in textures)
				t.Reload();
		}
	}
}