/*
 * Silverscreen
 * (C) 2010 William Lahti.
 *
 */

using System;
using System.IO;
using System.Text;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Reflection;
using System.Globalization;

using Tao.OpenGl;
using Tao.DevIl;
using Tao.Platform.X11;
using GtkGL;

namespace Silverscreen {
	public class Texture {
		public Texture(Texture t)
		{
			Manager = t.Manager;
			ScaleX = t.ScaleX;
			ScaleY = t.ScaleY;
			Filename = t.Filename;
			TextureHandle = t.TextureHandle;
			Width = t.Width;
			Height = t.Height;
			Owned = false;
		}

		public Texture(TextureManager manager, string filename)
		{
			Manager = manager;
			ScaleX = ScaleY = 1;
			Filename = filename;
			InvertedY = false;
			Owned = true;
			Reload ();
		}

		public Texture(TextureManager manager, int tex, int w, int h, bool owned)
		{
			Manager = manager;
			Width = w;
			Height = h;
			TextureHandle = tex;
			ScaleX = ScaleY = 1;
			InvertedY = false;
			Owned = owned;
		}

		public Texture(TextureManager manager, int tex, int w, int h):
			this (manager, tex, w, h, false)
		{
		}

		protected Texture(TextureManager manager)
		{
			Manager = manager;
			ScaleX = ScaleY = 1;
			InvertedY = false;
			Filename = null;
		}

		protected Texture():
			this ((TextureManager)null)
		{
		}

		protected void FreeTexture()
		{
			if (Owned && TextureHandle > 0) {
				uint tex = (uint)TextureHandle;
				Gl.glDeleteTextures (1, ref tex);
				TextureHandle = 0;
			}
		}

		protected virtual void Dispose (bool disposing)
		{
			if (disposing) GC.SuppressFinalize (this);
			FreeTexture();
		}

		public TextureManager Manager { get; protected set; }
		public string Filename { get; protected set; }
		public int Width { get; protected set; }
		public int Height { get; protected set; }
		public int Handle { get; protected set; }
		public int TextureHandle { get; protected set; }
		public double ScaleX { get; set; }
		public double ScaleY { get; set; }
		public bool InvertedY { get; set; }
		public bool Owned { get; private set; }

		float TexTop { get { return (InvertedY ? 0.0f : 1.0f); } }
		float TexBottom { get { return (InvertedY ? 1.0f : 0.0f); } }

		public void ForgetFilename ()
		{
			Filename = null;
		}

		public void Replace (int tex, int w, int h)
		{
			TextureHandle = tex;
			Width = w;
			Height = h;
		}

		public void Load (string filename)
		{
			if (filename == null)
				throw new ArgumentNullException ("filename");

			Filename = filename;
			Reload();
		}

		public void LoadForeign (TextureManager manager, string filename)
		{
			Filename = filename;
			manager.ScheduleReload (this);
		}

		public void Reload ()
		{
			Reload (true);
		}

		public void Reload (bool except)
		{
			try {

			if (Filename == null)
				throw new Exception ("No image loaded!");

			FreeTexture();

			int handle = Il.ilGenImage();

			Il.ilBindImage(handle);
			Il.ilLoadImage(Filename);
			ErrorCheck();
			Ilu.iluBuildMipmaps();
			int tex = Ilut.ilutGLBindMipmaps();

			glBind();
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MIN_FILTER, Gl.GL_LINEAR_MIPMAP_LINEAR);

			Handle = handle;
			Width = GetInteger(Il.IL_IMAGE_WIDTH);
			Height = GetInteger(Il.IL_IMAGE_HEIGHT);
			TextureHandle = tex;

			Il.ilDeleteImage (handle);

			} catch (Exception e) {
				if (except) throw;
				TextureHandle = 0;
				Width = 0;
				Height = 0;
			}
		}

		public override string ToString()
		{
			return string.Format("{0} ({1}x{2})", Path.GetFileName(Filename), Width, Height);
		}

		private int GetInteger(int cap)
		{
			int i = Il.ilGetInteger(cap);
			ErrorCheck();
			return i;
		}

		private void ErrorCheck()
		{
			int error;
			if ((error = Il.ilGetError()) != Il.IL_NO_ERROR) {
				if (error == Il.IL_COULD_NOT_OPEN_FILE)
					throw new FileNotFoundException(Filename);
				throw new Exception ("DevIL error " + error);
			}
		}

		public void ApplyScale (double scale)
		{
			ScaleX = ScaleY = scale;
		}

		public int ComputedWidth {
			get { return (int)(Width * ScaleX); }
		}

		public int ComputedHeight {
			get { return (int)(Height * ScaleY); }
		}

		public virtual void glBind()
		{
			Gl.glBindTexture(Gl.GL_TEXTURE_2D, TextureHandle);
		}

		public virtual void glUnbind()
		{
			Gl.glBindTexture(Gl.GL_TEXTURE_2D, 0);
		}

		public static Texture DelayLoad (TextureManager manager, string filename)
		{
			return DelayLoad (manager, filename, null);
		}

		public static Texture DelayLoad (TextureManager manager, string filename, Texture defaultTexture)
		{
			if (defaultTexture == null)
				defaultTexture = manager.BlankTexture;

			var tex = new Texture(defaultTexture);
			tex.Filename = filename;
			manager.ScheduleReload (tex);
			return tex;
		}

		public void RenderAspectScaled (float x, float y, float w, float h, bool maximize)
		{
			if (w == Width && h == Height) {
				Render (x, y);
				return;
			}

			if (w / Width == h / Height) {
				RenderScaled (x, y, w, h);
				return;
			}

			float scale = 1;
			int scalePlane = 0; // x = 0, y = 1

			if (maximize) {
				if (w / Width > h / Height)
					scalePlane = 0;
				else
					scalePlane = 1;
			} else {
				if (w / Width > h / Height || maximize)
					scalePlane = 1;
				else
					scalePlane = 0;
			}

			if (scalePlane == 1) {
				// scale by height
				scale = h / Height;
			} else {
				// scale by width
				scale = w / Width;
			}

			float iy = ((Height * scale - h) / 2);
			float ix = ((Width * scale - w) / 2);
			float wm = Math.Max (0, w - Width * scale);
			float hm = Math.Max (0, h - Height * scale);

			w -= wm;
			h -= hm;
			x += wm / 2;
			y += hm / 2;

			RenderRectangle(x, y, w, h, ix, iy, Width * (w / Width / scale), Height * (h / Height / scale));
		}

		public void Render(float x, float y)
		{
			glBind();
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_S, Gl.GL_CLAMP);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_T, Gl.GL_CLAMP);
			Gl.glBegin(Gl.GL_TRIANGLE_STRIP);
				Gl.glTexCoord2f(0.0f,TexTop); Gl.glVertex2f(x, y);
				Gl.glTexCoord2f(1.0f,TexTop); Gl.glVertex2f(x + ComputedWidth, y);
				Gl.glTexCoord2f(0.0f,TexBottom); Gl.glVertex2f(x, y + ComputedHeight);
				Gl.glTexCoord2f(1.0f,TexBottom); Gl.glVertex2f(x + ComputedWidth, y + ComputedHeight);
			Gl.glEnd();
			SilverGL.CheckGL();
			glUnbind();
		}

		[Obsolete("Use Render(x,y)")]
		public void glDrawAt2f(float x, float y)
		{
			Render (x, y);
		}

		/// <summary>
		/// Render the texture into the given rectangle, scaled to fit. This does NOT make use of or modify
		/// the ScaleX and ScaleY properties.
		/// </summary>
		public void RenderScaled(float x, float y, float w, float h)
		{
			RenderRectangle (x, y, w, h, 0, 0, Width, Height);
		}

		/// <summary>
		/// Render a subrectangle of the texture to the given rectangle on the screen.
		/// This method can be used to draw only a portion of a texture or to draw a texture
		/// scaled. Note that this method does NOT make use of or modify the ScaleX and ScaleY
		/// properties.
		/// </summary>
		public void RenderRectangle(float x, float y, float w, float h, float ix, float iy, float iw, float ih)
		{
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_S, Gl.GL_CLAMP);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_T, Gl.GL_CLAMP);
			float tx = ix / Width;
			float ty = iy / Height;
			float tw = iw / Width;
			float th = ih / Height;
			float texTop = 1.0f - ty;
			float texBottom = 1.0f - (ty + th);

			if (InvertedY) {
				texTop = ty;
				texBottom = ty + th;
			}

			glBind();
			Gl.glBegin(Gl.GL_TRIANGLE_STRIP);
				Gl.glTexCoord2f(tx,		 texTop);		Gl.glVertex2f(x, y);
				Gl.glTexCoord2f(tx + tw, texTop);		Gl.glVertex2f(x + w, y);
				Gl.glTexCoord2f(tx,		 texBottom);	Gl.glVertex2f(x, y + h);
				Gl.glTexCoord2f(tx + tw, texBottom);	Gl.glVertex2f(x + w, y + h);
			Gl.glEnd();
			glUnbind();
		}

		/// <summary>
		/// Pass values for w,h which are less than texture dimensions to
		/// achieve top-right anchored clipping. Pass values for w,h which
		/// are greater than texture dimensions to achieve texture repeating.
		/// </summary>
		public void RenderBox(float x, float y, float w, float h)
		{
			RenderBox (x, y, w, h, false);
		}

		public void RenderBox(float x, float y, float w, float h, bool clamp)
		{
			if (clamp) {
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_S, Gl.GL_CLAMP);
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_T, Gl.GL_CLAMP);
			} else {
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_S, Gl.GL_REPEAT);
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_T, Gl.GL_REPEAT);
			}
			float ty = h / ComputedHeight;
			float tx = w / ComputedWidth;
			float texTop = 1.0f;
			float texBottom = 1.0f - ty;

			if (InvertedY) {
				texTop = 0;
				texBottom = ty;
			}

			glBind();
			Gl.glBegin(Gl.GL_TRIANGLE_STRIP);
				Gl.glTexCoord2f(0.0f, texTop);		Gl.glVertex2f(x, y);
				Gl.glTexCoord2f(tx,   texTop);		Gl.glVertex2f(x + w, y);
				Gl.glTexCoord2f(0.0f, texBottom);	Gl.glVertex2f(x, y + h);
				Gl.glTexCoord2f(tx,   texBottom);	Gl.glVertex2f(x + w, y + h);
			Gl.glEnd();
			glUnbind();
		}
	}
}