/*
 * Silverscreen
 * (C) 2010 William Lahti.
 *
 */

using System;
using System.IO;
using System.Threading;
using System.Runtime.InteropServices;
using System.Reflection;
using System.Globalization;
using System.Collections.Generic;

using Tao.OpenGl;
using Tao.DevIl;
using Tao.Sdl;
using GtkGL;

namespace Silverscreen {
	public class GLVideoSink : Gst.Bin {
		public GLVideoSink (Application app, bool redirectRendering, bool useVideo)
		{
			Application = app;
			
			Gst.CorePlugins.FakeSink fsink;
			Sink = fsink = Gst.ElementFactory.Make("fakesink", "vsink") as Gst.CorePlugins.FakeSink;
			//Sink["max-lateness"] = 250000000L; // a quarter second (ns)
			Add (Sink);
			
			if (useVideo) {
				Console.WriteLine ("Initializing GLUpload...");

				Sink["signal-handoffs"] = true;
				fsink.Handoff += OnHandoff;

				GLUpload = Gst.ElementFactory.Make("glupload", "opengl-adaptor");
				Console.WriteLine ("sending external GL context: {0}", app.GLContext.GLXContext);
				GLUpload["external-opengl-context"] = (uint)app.GLContext.GLXContext;
				
				Add (GLUpload);
				GLUpload.Link (Sink);
				AddPad (new Gst.GhostPad ("sink0", GLUpload.GetStaticPad("sink")));
				
				Console.WriteLine ("...done");
			} else {
				AddPad (new Gst.GhostPad ("sink0", fsink.GetStaticPad("sink")));
			}
		}
		
		private delegate void DrawCallback (uint tex, int w, int h);

		public bool HasVideo { get { return GLUpload != null; } }
		public Application Application { get; private set; }
		public bool Threaded { get; private set; }
		public Texture Texture { get; private set; }
		public Gst.Element GLUpload { get; private set; }
		public Gst.Element Sink { get; private set; }
		public IntPtr GstGLDisplay { get; private set; }

		[DllImport("libgstgl-0.10.so")] private static extern void gst_gl_display_lock (IntPtr dpy);
		[DllImport("libgstgl-0.10.so")] private static extern void gst_gl_display_unlock (IntPtr dpy);

		bool locked = false;

		public void Start ()
		{
			if (GLUpload == null) return;
			// Wait for the first frame to arrive.
			Console.WriteLine ("GLVideoSink: waiting for first frame... (locked: {0})", locked);
			while (Texture == null) {
				Thread.Sleep (0);
				Gtk.Application.RunIteration(false);
			}

			Console.WriteLine ("Got first frame! Go!");
			
		}
		
		public void LockSync ()
		{
			if (GLUpload == null)
				return;
				
			if (locked)
				throw new InvalidOperationException ("GstGLDisplay is already locked! Without safeguard the application will deadlock");
			locked = true;

			if (GstGLDisplay != IntPtr.Zero)
				gst_gl_display_lock (GstGLDisplay);
		}

		public void UnlockSync ()
		{
			if (GLUpload == null)
				return;
				
			if (!locked) {
				Console.Error.WriteLine("WARNING: Trying to unlock, but GstGLDisplay is not locked!");
				return;
			}
			
			locked = false;

			if (GstGLDisplay != IntPtr.Zero)
				gst_gl_display_unlock (GstGLDisplay);
		}

		internal void LockThread ()
		{
			if (GstGLDisplay != IntPtr.Zero)
				gst_gl_display_lock (GstGLDisplay);
		}

		internal void UnlockThread ()
		{
			if (GstGLDisplay != IntPtr.Zero)
				gst_gl_display_unlock (GstGLDisplay);
		}


		Gst.Buffer lastBuffer;
		List<Gst.Buffer> buffers = new List<Gst.Buffer>();
		int maxBuffers = 2;
		
		private void AddBuffer (Gst.Buffer buf)
		{
			// Dispose *BEFORE* locking because buffer destructor locks the gl thread

// 			w = buf.GLGetWidth ();
// 			h = buf.GLGetHeight ();
// 			return (int)buf.GLGetTexture ();

			buffers.Add (buf);

			int index = Math.Min (buffers.Count - 1, 2);
			
			if (buffers.Count > maxBuffers && index == 0) {
				index = 1;
			}

			var rbuf = buffers[index];
			int w = rbuf.GLGetWidth();
			int h = rbuf.GLGetHeight();
			int tex = (int)rbuf.GLGetTexture ();

			if (Texture == null)
				Texture = new Texture (Application.Textures, tex, w, h);
			else
				Texture.Replace (tex, w, h);

			if (buffers.Count > maxBuffers) {
				buffers[0].Dispose();
				buffers.RemoveAt(0);
			}
		}
		
		private void OnHandoff (object o, Gst.CorePlugins.FakeSink.HandoffArgs e)
		{
			if (GLUpload == null)
				return;

			if (e.Buffer == null)
				return;

			GstGLDisplay = e.Buffer.GLGetDisplay ();
			AddBuffer (e.Buffer);
		}

		[DllImport("libgstreamer-0.10.so", EntryPoint = "gst_mini_object_ref")]
			private static extern IntPtr  gst_buffer_ref(IntPtr buf);
		//[DllImport("libgstreamer-0.10.so", EntryPoint = "gst_mini_object_unref")]
		//	private static extern void gst_buffer_unref (IntPtr buf);
	}
}