/*
 * Silverscreen
 * (C) 2010 William Lahti.
 *
 */

using System;
using System.IO;
using System.Text;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Reflection;
using System.Globalization;

using Tao.OpenGl;
using Tao.DevIl;
using Tao.Platform.X11;
using GtkGL;

namespace Silverscreen {
	public struct XVisualInfo {
		public IntPtr visual;
		public int visualid;
		public int screen;
		public int depth;
		public int _class;
		public uint red_mask;
		public uint green_mask;
		public uint blue_mask;
		public int colormap_size;
		public int bits_per_rgb;
	}

	enum XVisualInfoMask: uint {
		None =             0,
		ID =               0x1,
		Screen =           0x2,
		Depth =            0x4,
		Class =            0x8,
		RedMask =          0x10,
		GreenMask =        0x20,
		BlueMask =         0x40,
		ColormapSize =     0x80,
		BitsPerRGB =       0x100,
		All =              0x1ff,
	}

	public unsafe class Context {
		public Context (Gdk.Display dpy, bool directRendering, params int[] attributes)
		{
			this.attributes = attributes;
			this.dpy = dpy;
			xdpy = gdk_x11_display_get_xdisplay (dpy.Handle);
			handle = gdk_gl_context_attrlist_share_new(attributes, IntPtr.Zero, (directRendering ? 1 : 0));

			if (handle == IntPtr.Zero)
				throw new Exception ("Failed to create GDK OpenGL context!");
		}

		Gdk.Display dpy;
		IntPtr xdpy;
		IntPtr handle;
		Gdk.Drawable currentDrawable;
		IntPtr glxContext = IntPtr.Zero;
		int[] attributes;

		public Gdk.Display Display { get { return dpy; } }
		public IntPtr GLXContext { get { return glxContext; } }

		public enum Attributes : int {
			None						= 0,
			UseGL						= 1,
			BufferSize					= 2,
			Level						= 3,
			RGBA						= 4,
			DoubleBuffer				= 5,
			Stereo						= 6,
			AuxBuffers					= 7,
			RedSize						= 8,
			GreenSize					= 9,
			BlueSize					= 10,
			AlphaSize					= 11,
			DepthSize					= 12,
			StencilSize					= 13,
			AccumRedSize				= 14,
			AccumGreenSize				= 15,
			AccumBlueSize				= 16,
			AccumAlphaSize				= 17,

			/* GLX_EXT_visual_info extension */
			XVisualTypeExt				= 0x22,
			TransparentTypeExt			= 0x23,
			TransparentIndexValueExt	= 0x24,
			TransparentRedValueExt		= 0x25,
			TransparentGreenValueExt	= 0x26,
			TransparentBlueValueExt		= 0x27,
			TransparentAlphaValueExt	= 0x28
		}

		public void WaitGL ()
		{
			gdk_gl_wait_gl();
		}

		public void Disable ()
		{
			currentDrawable = null;
			glXMakeCurrent (xdpy, 0, IntPtr.Zero);
			//gdk_gl_make_current (IntPtr.Zero, this.handle);
		}

		bool inited = false;

		public void MakeCurrent (Gdk.Drawable drawable)
		{
			if (drawable == null)
				throw new ArgumentNullException("drawable");
			currentDrawable = drawable;
			gdk_gl_make_current (drawable.Handle, this.handle);
			if (glxContext == IntPtr.Zero)
				glxContext = glXGetCurrentContext();

			if (!inited) {
				int screen;
				glXQueryContext(xdpy, glxContext, 0x800C /*GLX_SCREEN*/, out screen);
				IntPtr vis = glXChooseVisual (xdpy, screen, attributes);
				int id = XVisualIDFromVisual (vis);
				XVisualInfo template = new XVisualInfo();
				template.visualid = id;
				XVisualInfo[] matches = XGetVisualInfo (xdpy, (int)XVisualInfoMask.ID, ref template);
				Console.WriteLine ("chose visual: {0} (aka 0x{1})", id, ((int)vis).ToString("x"));
				foreach (XVisualInfo info in matches) {
					Console.WriteLine (" - match {0} (aka 0x{1})", info.visualid, ((int)info.visual).ToString("x"));
					Console.WriteLine ("   * screen: {0}", info.screen);
					Console.WriteLine ("   * depth: {0}", info.depth);
					Console.WriteLine ("   * class: {0}", info._class);
					Console.WriteLine ("   * mask: {0}/{1}/{2}", info.red_mask.ToString("x"), info.green_mask.ToString("x"),
						info.blue_mask.ToString("x"));
					Console.WriteLine ("   * colormap size: {0}", info.colormap_size);
					Console.WriteLine ("   * bits per rgb: {0}", info.bits_per_rgb);
				}

				inited = true;
			}
		}

		public void SwapBuffers ()
		{
			if (currentDrawable == null)
				throw new InvalidOperationException ("Must call MakeCurrent() first");

			gdk_gl_swap_buffers (currentDrawable.Handle);
		}

		public static Gdk.Visual ChooseVisual (params int[] attr)
		{
			int[] attr2 = new int[attr.Length];

			for (int x = 0; x < attr2.Length; ++x)
				attr2[x] = (int)attr[x];
			IntPtr vis = gdk_gl_choose_visual (attr2);

			if (vis == IntPtr.Zero)
				return null;
			else
				return new Gdk.Visual(vis);
		}

		public const string GtkGLDll = "libgtkgl-2.0.so";
		public const string GdkX11Dll = "libgdk-x11-2.0.so";
		public const string X11Dll = "libX11.so.6";
		public const string XCompositeDll = "libXcomposite.so.1";
		public const string GlDll = "libGL.so";
		[DllImport(GtkGLDll)]
			public static extern IntPtr gdk_gl_choose_visual ([MarshalAs(UnmanagedType.LPArray)] int[] attrlist);
		[DllImport(GtkGLDll)]
			public static extern int gdk_gl_make_current  (IntPtr drawable, IntPtr context);
		[DllImport(GtkGLDll)]
			public static extern void gdk_gl_swap_buffers  (IntPtr drawable);
		[DllImport(GtkGLDll)]
			public static extern void gdk_gl_wait_gl ();
		[DllImport(GtkGLDll)]
			public static extern IntPtr gdk_gl_pixmap_new (IntPtr gdkVisual, IntPtr gdkPixmap);
		[DllImport(GtkGLDll)]
			public static extern IntPtr gdk_gl_pixmap_make_current(IntPtr gdkglPixmap, IntPtr gdkglContext);
		[DllImport(GtkGLDll)]
			public static extern IntPtr gdk_gl_context_attrlist_share_new ([MarshalAs(UnmanagedType.LPArray)] int[] attrlist,
					IntPtr shareList, int direct);
		[DllImport(GdkX11Dll)]
			public static extern IntPtr gdk_x11_display_get_xdisplay (IntPtr gdkDisplay);
		[DllImport(GdkX11Dll)]
			public static extern int gdk_x11_screen_get_screen_number (IntPtr gdkScreen);
		[DllImport(GdkX11Dll)]
			public static extern IntPtr gdk_x11_visual_get_xvisual (IntPtr gdkVisual);
		[DllImport(GdkX11Dll)]
			public static extern int gdk_x11_drawable_get_xid (IntPtr gdkDrawable);
		[DllImport(GlDll, EntryPoint = "glXGetClientString")]
			private static extern byte *_glXGetClientString (IntPtr xdpy, int name);
		[DllImport(GlDll)]
			private static extern IntPtr glXGetCurrentContext ();
		[DllImport(GlDll)]
			private static extern IntPtr glXChooseVisual (IntPtr xdpy, int screen, int[] attribList);
		[DllImport(GlDll)]
			private static extern IntPtr glXQueryContext (IntPtr xdpy, IntPtr ctx, int attrib, out int value);
		[DllImport(GlDll)]
			private static extern void glXMakeCurrent (IntPtr xdpy, int xdrawable, IntPtr context);

		public static unsafe string glXGetClientString (IntPtr xdpy, int name)
		{
			// Tao's version of this function assumes client frees string, but the
			// string is statically allocated which results in a crash. So here we
			// do it right.
			int len = 0;
			byte *bytes = _glXGetClientString(xdpy, name);

			if (bytes == null)
				return null;

			for (byte *bp = bytes; *bp != 0; ++bp) ++len;
			byte[] arr = new byte[len];
			for (int x = 0; x < len; ++x) arr[x] = bytes[x];
			return Encoding.ASCII.GetString (arr);
		}

		[DllImport(X11Dll)]
			public static extern int XVisualIDFromVisual (IntPtr vis);

		[DllImport(X11Dll)]
			public static extern IntPtr XGetVisualInfo(IntPtr display, int vinfo_mask, ref XVisualInfo vinfo_template, out int nitems_return);

		public unsafe static XVisualInfo[] XGetVisualInfo (IntPtr display, int mask, ref XVisualInfo template)
		{
			int items;
			IntPtr *result = (IntPtr*)XGetVisualInfo (display, mask, ref template, out items);
			XVisualInfo[] arr = new XVisualInfo[items];

			for (int x = 0; x < items; ++x) {
				arr[x] = (XVisualInfo)Marshal.PtrToStructure (*result, typeof (XVisualInfo));
				++result;
			}

			return arr;
		}

		[DllImport(X11Dll)]
			public static extern void XLockDisplay (IntPtr dpy);
		[DllImport(X11Dll)]
			public static extern void XUnlockDisplay (IntPtr dpy);
		[DllImport(X11Dll)]
			public static extern int XFlush (IntPtr dpy);
		[DllImport(X11Dll)]
			public static extern int XSync (IntPtr dpy, bool discard);

		[DllImport(X11Dll)]
			public static extern int XFreePixmap (IntPtr dpy, int pixmap);

		[DllImport(XCompositeDll)]
			public static extern void XCompositeRedirectWindow (IntPtr dpy, int window, int update);
		[DllImport(XCompositeDll)]
			public static extern int XCompositeNameWindowPixmap (IntPtr dpy, int window);

		private static glXBindTexImageEXTProto _glXBindTexImageEXT;
		private delegate void glXBindTexImageEXTProto (IntPtr dpy, int glxDrawable, int buffer, int[] attriblist);
		public static void glXBindTexImageEXT (IntPtr dpy, int glxDrawable, int buffer, int[] attrs)
		{
			if (_glXBindTexImageEXT == null)
				_glXBindTexImageEXT = Marshal.GetDelegateForFunctionPointer (Glx.glxGetProcAddress("glXBindTexImageEXT"),
					typeof (glXBindTexImageEXTProto)) as glXBindTexImageEXTProto;
			if (_glXBindTexImageEXT == null)
				throw new Exception ("Could not get address of glXBindTexImageEXT method");

			_glXBindTexImageEXT (dpy, glxDrawable, buffer, attrs);
		}

		public static unsafe IntPtr[] glXGetFBConfigs (IntPtr dpy, int screen)
		{
			int elements;
			IntPtr arr = Glx.glXGetFBConfigs(dpy, screen, &elements);

			if (elements == 0)
				return new IntPtr[0];

			IntPtr[] ret = new IntPtr[elements];

			IntPtr *basep = (IntPtr*)arr;
			for (int x = 0; x < ret.Length; ++x) {
				ret[x] = *basep;
				++basep;
			}

			return ret;
		}

		public static IntPtr GetXHandle (Gdk.Display dpy)
		{
			return gdk_x11_display_get_xdisplay (dpy.Handle);
		}

		public static int GetXHandle (Gdk.Drawable dr)
		{
			return gdk_x11_drawable_get_xid (dr.Handle);
		}
		
	}
}