/*
 * Silverscreen
 * (C) 2010 William Lahti.
 *
 */

using System;
using System.IO;
using System.Text;
using System.Diagnostics;
using System.Security.Cryptography;
using System.Threading;
using System.Runtime.InteropServices;
using System.Reflection;
using System.Globalization;
using System.Collections.Generic;

using Tao.OpenGl;
using Tao.DevIl;
using GtkGL;

namespace Silverscreen {
	public delegate void Block();

	public class Application {
		public bool LockDebug = true;

		public Application(string[] args)
		{
			DisplayInit (args);

			Textures = new TextureManager ();

			if (args.Length > 0)
				MediaFile = args[0];

			Thread.CurrentThread.Priority = ThreadPriority.BelowNormal;
		}

		public TextureManager Textures { get; private set; }
		public string MediaFile;

		bool exited = false;
		Gdk.Display display;
		Gdk.Screen[] screens;
		Context context;
		Window window;
		bool moreEvents = false;
		int w, h;

		public Random Random = new Random();
		public Context GLContext { get { return context; } }
		public int WindowWidth { get { return w; } }
		public int WindowHeight { get { return h; } }
		public Window			Window { get { return window; } }
		public Gdk.Display		Display { get { return display; } }
		public Gdk.Screen[]		Screens { get { return screens; } }
		public bool				IsFullscreen = false;

		#region Initialization / Shutdown

		public static int Main(string[] args)
		{
			Console.WriteLine ("Starting GLVideoTest...");
			return new Application(args).Run();
		}

		public void GLLockDisplay ()
		{
			Context.XLockDisplay (Context.GetXHandle (display));
		}

		public void GLUnlockDisplay ()
		{
			Context.XUnlockDisplay (Context.GetXHandle (display));
		}

		private void DisplayInit(string[] args)
		{
			Gtk.Application.Init("Silverscreen", ref args);
			Gst.Application.Init();

			display = Gdk.Display.Default;
			screens = new Gdk.Screen[display.NScreens];

			for (int x = 0; x < display.NScreens; ++x)
				screens[x] = display.GetScreen(x);

			context = new Context(display, true,
				(int)Context.Attributes.RGBA,
				(int)Context.Attributes.DoubleBuffer,
				(int)Context.Attributes.RedSize, 1,
				(int)Context.Attributes.GreenSize, 1,
				(int)Context.Attributes.BlueSize, 1,
				//(int)Context.Attributes.StencilSize, 1,
				0
			);

			Gtk.Application.RunIteration(false);

		}

		public void Exit ()
		{
			exited = true;
		}

		#endregion
		#region Event Handlers
		#region  - OnWindowState

		public void OnWindowState(object sender, Gtk.WindowStateEventArgs e)
		{
			if ((e.Event.NewWindowState & Gdk.WindowState.Maximized) != 0) {
				window.Unmaximize();
				Fullscreen();
			}
		}

		#endregion
		#region Display Controls

		public void Fullscreen ()
		{
			IsFullscreen = true;
			window.Fullscreen();
		}

		public void Unfullscreen()
		{
			IsFullscreen = false;
			window.Unfullscreen();
		}

		#endregion

		private void UpdateAspects()
		{
			int ww, wh;
			window.GetSize (out ww, out wh);
			context.MakeCurrent(window.GdkWindow);

			Gl.glViewport(0, 0, ww, wh);
			Gl.glMatrixMode(Gl.GL_PROJECTION);
			Gl.glLoadIdentity();
			//Glu.gluPerspective(0, ww / (float)wh, 0.1f, 30.0f);
			Gl.glOrtho(0, ww, wh, 0, 0, 1);
			Gl.glMatrixMode(Gl.GL_MODELVIEW);
			Gl.glLoadIdentity();

			ApplyGLConfig ();

		}

		#endregion

		public void GLContextMakeCurrent ()
		{
			context.MakeCurrent(window.GdkWindow);
		}

		public void GLContextDisable ()
		{
			context.Disable();
		}

		public bool Render ()
		{
			Gl.glColor4f(0,0,0,1);
			SilverGL.Quad(0, 0, WindowWidth, WindowHeight, false);

			var tex = vsink.Texture;

			if (tex != null && tex.TextureHandle != 0) {
				//if (!Gl.glIsTexture ((uint)tex.TextureHandle))
				//	throw new Exception ("This thing is not a texture.");
				/*
				gst_gl_display_redisplay((Application.Media.VideoSink as GLVideoSink).GstGLDisplay,
					(uint)tex.TextureHandle, tex.Width, tex.Height, false);
				/*/
				Console.WriteLine ("rendering tex {0}", tex.TextureHandle);
				Gl.glUseProgramObjectARB (0);
				Gl.glBindTexture (Gl.GL_TEXTURE_2D, 0);
				Gl.glEnable (Gl.GL_TEXTURE_RECTANGLE_ARB);
				Gl.glBindTexture (Gl.GL_TEXTURE_RECTANGLE_ARB, (uint)tex.TextureHandle);

				SilverGL.White ();
				RenderFrame (0, 0, WindowWidth, WindowHeight, 0, 0, tex.Width / 2, tex.Height / 2);
				Gl.glBindTexture (Gl.GL_TEXTURE_RECTANGLE_ARB, 0);
			}

			return true;
		}

		private void RenderFrame (float x, float y, float w, float h, int tx, int ty, int tw, int th)
		{
			Gl.glFlush ();
			Gl.glBegin(Gl.GL_TRIANGLE_STRIP);
				Gl.glTexCoord2i(tx,			ty); Gl.glVertex2f(x, y);
				Gl.glTexCoord2i(tx + tw,	ty); Gl.glVertex2f(x + w, y);
				Gl.glTexCoord2i(tx,			ty + th); Gl.glVertex2f(x, y + h);
				Gl.glTexCoord2i(tx + tw,	ty + th); Gl.glVertex2f(x + w, y + h);
			Gl.glEnd();
		}

		private void ApplyGLConfig ()
		{
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_FASTEST);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
			Gl.glEnable(Gl.GL_TEXTURE_2D);
			Gl.glEnable(Gl.GL_TEXTURE_RECTANGLE_ARB);
			Gl.glEnable(Gl.GL_BLEND);
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			Gl.glBlendFunc (Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
			Gl.glDisable (Gl.GL_DITHER);
		}

		Texture splash = null;

		public float SplashProgress = 0;
		public string SplashTask = "Please wait...";
		public bool Loading = false;
		public float LoadingAlpha = 0;
		public string LoadingTask = string.Empty;
		public float LoadingProgress = 0;
		public Gst.Pipeline pipeline;
		public GLVideoSink vsink;
		public Gst.Element asink;

		public bool MediaBusWatch (Gst.Pipeline pipe, Gst.Bus bus, Gst.Message message)
		{
			switch (message.Type) {
			case Gst.MessageType.Error:
				string msg;
				Enum err;
				message.ParseError (out err, out msg);

				Console.WriteLine ("Gstreamer error: {0}", msg);
				break;
			case Gst.MessageType.Eos:
				Environment.Exit (0);
				break;
			case Gst.MessageType.StateChanged:
			case Gst.MessageType.NewClock:
			case Gst.MessageType.AsyncDone:
			case Gst.MessageType.StreamStatus:
				break;
			case Gst.MessageType.Tag:
				Gst.Pad pad;
				Gst.TagList tags;
				message.ParseTag (out pad, out tags);
				foreach (string tag in tags.Tags) {
					Console.WriteLine ("Tag {0} = '{1}'", tag, tags[tag]);
				}

				break;
			case Gst.MessageType.Warning:
				Enum code;
				string text;
				message.ParseWarning (out code, out text);
				Console.WriteLine ("Gstreamer Warning ({0}): {1}", code.ToString(), text);
				break;
			default:
				Console.WriteLine ("Unknown gstreamer bus message:\t{0}", message.Type);
				break;
			}

			return true;
		}

		public int Run()
		{

			window = new Window ("GL Video Test");
			window.Realize();

			context.MakeCurrent(window.GdkWindow);

			var playbin = Gst.ElementFactory.Make ("playbin2", "player") as Gst.Pipeline;
			playbin["video-sink"] = vsink = new GLVideoSink (this, true, true);
			playbin["audio-sink"] = (asink = Gst.ElementFactory.Make ("pulsesink", "audiosink"));
			playbin["uri"] = MediaFile;

			playbin.Bus.AddWatch(new Gst.BusFunc (
				delegate (Gst.Bus bus, Gst.Message message) {
					return MediaBusWatch (playbin, bus, message);
				}
			));

			ApplyGLConfig ();

			Il.ilInit();
			Ilu.iluInit();
			Ilut.ilutRenderer(Ilut.ILUT_OPENGL);

			w = h = 0;

			window.ResizeChecked += delegate (object sender, EventArgs e)
			{
				int ww, wh;

				window.GetSize(out ww, out wh);
				if (ww != w || wh != h) {
					UpdateAspects();
					w = ww;
					h = wh;
				}
			};

			{
				int ww = w, wh = h;
				window.GetSize(out ww, out wh);
				if (ww != w || wh != h) {
					UpdateAspects();
					w = ww;
					h = wh;
				}
			}

			window.Show ();
			playbin.Play();

			while (true) {
				SilverGL.CheckGL ();

				vsink.LockSync();
				GLContextMakeCurrent ();
				ApplyGLConfig ();

				if (!Render ()) break;

				SilverGL.CheckGL ();
				context.SwapBuffers();

				Gl.glUseProgramObjectARB (0);
				Gl.glBindTexture (Gl.GL_TEXTURE_2D, 0);
				GLContextDisable();

				vsink.UnlockSync ();

				do {
					moreEvents = false;
					Gtk.Application.RunIteration (false);
				} while (moreEvents);

				Thread.Sleep (0);
			}

			return 0;
		}

		protected void OnDelete (object sender, Gtk.DeleteEventArgs e)
		{
		}
	}
}