/* ========================================================================= *
 * File: fpumode.c
 *
 * Copyright (C) 2010 Nokia Corporation. All rights reserved.
 *
 * Contacts: Leonid Moiseichuk <leonid.moiseichuk@nokia.com>
 *
 *  This preload library can be compiled in 3 versions:
 *  - default -> show current FPU control status and sets _FPU_DEFAULT
 *  - -fast -> select RUN FAST mode for ARM processors
 *  - -ieee -> select IEEE mode for ARM processors
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

/* ========================================================================= *
 * Includes
 * ========================================================================= */

#include <stdio.h>
#include <fpu_control.h>

/* ========================================================================= *
 * Local definitions.
 * ========================================================================= */

#ifdef __arm__
#define _CW_RUNFAST_MASK        (3 << 24)
#define _CW_RUNFAST_MODE(cw)    ((cw | _CW_RUNFAST_MASK) & (~ _FPU_IEEE))
#define _CW_IEEE_MODE(cw)       ((cw & (~_CW_RUNFAST_MASK)) | _FPU_IEEE)
#endif

#define OUT 2   /* file handler to use */

/* ========================================================================= *
 * Function which performs all required actions.
 * ========================================================================= */

/* ------------------------------------------------------------------------- *
 * fpumode -- shows current fpu mode and if pointed - updates to specified.
 * parameters:
 *    nothing.
 * returns:
 *    nothing.
 * ------------------------------------------------------------------------- */

static void __attribute__((constructor))
fpumode(void)
{
  fpu_control_t cw = 0;

  _FPU_GETCW(cw);
  dprintf (OUT, "* fpu mode build %s %s\n", __DATE__, __TIME__);
  dprintf (OUT, "* current fpu mode is 0x%08x [%s]\n",
            (unsigned)cw,
#ifdef __arm__
          (_CW_RUNFAST_MODE(cw) == cw ? "RUN FAST" : (_FPU_IEEE == cw ? "IEEE" : "CUSTOM"))
#else
          (_FPU_DEFAULT == cw ? "DEFAULT" : (_FPU_IEEE == cw ? "IEEE" : "CUSTOM"))
#endif
      );

#if   defined(_IEEE_)
  cw = _CW_IEEE_MODE(cw);
  dprintf (OUT, "* changing mode to 0x%08x [IEEE]\n", cw);
#elif defined(_FAST_) && defined(__arm__)
  cw = _CW_RUNFAST_MODE(cw);
  dprintf (OUT, "* changing mode to 0x%08x [RUN FAST]\n", cw);
#else
  cw = _FPU_DEFAULT;
  dprintf (OUT, "* changing mode to 0x%08x [DEFAULT]\n", cw);
#endif
  _FPU_SETCW(cw);
} /* fpumode */

/* ===========================[ end of file fpumode.c ]======================= */
