/*
 * The source code in this file can be freely used, adapted,
 * and redistributed in source or binary form, so long as an
 * acknowledgment appears in derived source files.  The citation
 * should list that the code comes from the book "Linux Device
 * Drivers" by Alessandro Rubini and Jonathan Corbet, published
 * by O'Reilly & Associates.   No warranty is attached;
 * we cannot take responsibility for errors or fitness for use.
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>   /* printk() */
#include <linux/slab.h>     /* kmalloc() */
#include <linux/fs.h>       /* everything... */
#include <linux/errno.h>    /* error codes */
#include <linux/types.h>    /* size_t */
#include <linux/mm.h>
#include <linux/kdev_t.h>
#include <asm/page.h>
#include <linux/cdev.h>

#include <asm/io.h>
#include <linux/device.h>

#include <asm/pgtable.h>

//*****************************************************************************

MODULE_LICENSE("Dual BSD/GPL");

//*****************************************************************************

#define ksDriverName "FPGArw"

#define kMyMajorNum (251)
#define kMyMinorNum (11)

static int FPGA_Major = kMyMajorNum;
static int FPGA_Minor = kMyMinorNum;

//*****************************************************************************

// Map a 64-KB region starting an unmanged memory 0x0FB00000 ...
//
#define kFPGA_RegionStart    (0x0FB00000)
#define kFPGA_RegionSize     (0x00010000)
#define kFPGA_RegionPages    (kFPGA_RegionSize >> PAGE_SHIFT)
#define kFPGA_RegionEnd      (kFPGA_RegionStart + kFPGA_RegionSize - 1)

static unsigned long FPGA_RegionStart = (kFPGA_RegionStart);
static unsigned long FPGA_RegionSize  = (kFPGA_RegionSize);
static unsigned long FPGA_RegionPages = (kFPGA_RegionPages);
static unsigned long FPGA_RegionEnd   = (kFPGA_RegionEnd);

//*****************************************************************************

static int major = kMyMajorNum;
module_param(major, int, 0);

static int minor = kMyMinorNum;
module_param(minor, int, 0);

static unsigned long start = kFPGA_RegionStart;
module_param(start, ulong, 0);

static unsigned long size = kFPGA_RegionSize;
module_param(size,  ulong, 0);

//*****************************************************************************

static unsigned long FPGA_bIsOpen = 0;

static u32 *FPGA_pRegs = NULL;

//*****************************************************************************

/*
 * Common VMA ops.
 */

void FPGA_vma_open(struct vm_area_struct *vma)
{
    printk(KERN_NOTICE "%s VMA open, virt %lx - %lx, phys %lx\n",
            ksDriverName, vma->vm_start, vma->vm_end, vma->vm_pgoff << PAGE_SHIFT);
}

void FPGA_vma_close(struct vm_area_struct *vma)
{
    printk(KERN_NOTICE "%s VMA close.\n", ksDriverName);
}

//*****************************************************************************

static struct vm_operations_struct FPGA_remap_vm_ops =
{
    .open   = FPGA_vma_open,
    .close  = FPGA_vma_close,
};

//*****************************************************************************

#if 0

static int FPGA_mmap(struct file *filp, struct vm_area_struct *vma)
{
    unsigned long size = vma->vm_end - vma->vm_start;

    if (vma->vm_pgoff > FPGA_RegionPages)    return -EINVAL;

printk("FPGArw_mmap: pgoff = %lx, size=%lx \n", vma->vm_pgoff, size);

    vma->vm_start = (unsigned long)FPGA_RegionStart + (vma->vm_pgoff << PAGE_SHIFT);
    vma->vm_end   = (unsigned long)vma->vm_start + size;

printk("FPGArw_mmap: start = %lx, end=%lx \n", vma->vm_start, vma->vm_end);

    vma->vm_pgoff = ((vma->vm_pgoff << PAGE_SHIFT)
                      +  (unsigned long)FPGA_RegionStart)
                      >> PAGE_SHIFT;

    vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);

    vma->vm_page_prot |= PAGE_SHARED;

    vma->vm_flags |= (VM_IO | VM_RESERVED | VM_MAYSHARE | VM_SHARED);
    vma->vm_flags |= (VM_IO | VM_RESERVED | VM_READ     | VM_WRITE | VM_MAYREAD | VM_MAYWRITE);

    if (remap_pfn_range(vma,
                        vma->vm_start,
                        vma->vm_pgoff,
                        vma->vm_end - vma->vm_start,
                        vma->vm_page_prot))
        return -EAGAIN;

    vma->vm_ops = &FPGA_remap_vm_ops;
    FPGA_vma_open(vma);

    return 0;
}

#else

//*****************************************************************************

/*
 * A remap_pfn_range version of mmap.  This one is heavily borrowed
 * from drivers/char/mem.c.
 * Includes errata from LDD3 & the PFN stuff, plus my own fixes. -- RJE
 *
 */
static int FPGA_mmap2(struct file *filp, struct vm_area_struct *vma)
{
    unsigned long vsize  = vma->vm_end - vma->vm_start;
    unsigned long offset = (vma->vm_pgoff << PAGE_SHIFT);
    unsigned long psize  = FPGA_RegionSize - offset;
    unsigned long pfn    = ((unsigned long)FPGA_RegionStart + offset) >> PAGE_SHIFT;

printk("FPGArw_mmap2: pgoff = %lx, vm_start=%lx vm_end=%lx\n", vma->vm_pgoff, vma->vm_start, vma->vm_end);

printk("FPGArw_mmap2 pgoff = %lx,     size=%lx \n", vma->vm_pgoff, vsize);
printk("FPGArw_mmap2:  pfn = %lx, pfn_size=%lx \n", pfn, psize );

    if (vma->vm_pgoff > FPGA_RegionPages)
    {
        printk("FPGA_mmap2: inital offset (from MMAP) OOB!\n");
        return -EINVAL;
    }

    if (vsize > psize)
    {
        printk("FPGA_mmap2: virtual size exceeds remaining physical region!\n");
        return -EINVAL;
    }

    vma->vm_pgoff = (pfn);

    //
    // This is from /dev/mem ...
    //
    vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);

    // This is done in some of the examples ...
    //
    vma->vm_page_prot |= PAGE_SHARED;

    vma->vm_flags |= (VM_IO | VM_RESERVED);

    if (remap_pfn_range(vma, vma->vm_start, pfn, vsize, vma->vm_page_prot))
    {
        printk("FPGA_mmap2: remapping device PFN FAILED!\n");
        return -EAGAIN;
    }

    vma->vm_ops = &FPGA_remap_vm_ops;
    FPGA_vma_open(vma);

    return 0;
}

#endif

//#############################################################################

static ssize_t
FPGArw_read(struct file *pFile_io,
                 char __user *pUserBuf_io,
                 size_t       nCount_i,
                 loff_t      *pFPos_o)
{
#define kFPGA_MinReadSize  (sizeof(u32))
#define kFPGA_AlignSize    (sizeof(u32))

    //loff_t nPos           = *pFPos_o;
    unsigned long nPos    = *pFPos_o;
    int    nDeviceSize    = FPGA_RegionSize;
    size_t nAdjustedCount = nCount_i;

printk(KERN_INFO "%s READ(%p, %x, %08x) \n", ksDriverName, pUserBuf_io, nCount_i, nPos);

    // check limits ...
    if (nPos >= nDeviceSize)               return -EINVAL;
    if ((nPos + nCount_i) > nDeviceSize)   nAdjustedCount = nDeviceSize - nPos;

printk("    Adjusted vals [device] : nPos=%08lx,Count=%08lx\n", nPos, nAdjustedCount);

    if(nAdjustedCount < kFPGA_MinReadSize)
    {
        printk(KERN_WARNING "%s adjusted read count (%d) too small!\n", ksDriverName, nAdjustedCount);
        return -EINVAL;
    }

    if(nPos % kFPGA_AlignSize)
    {
        printk(KERN_WARNING "%s: read misaligned!\n", ksDriverName);
        return -EINVAL;
    }

    u32 __user *pBuf32   = (u32 __user *)pUserBuf_io;
    u32        *pFPGA    = (u32 *)FPGA_pRegs + (nPos/sizeof(u32));
    u32         nCount32 = nAdjustedCount/sizeof(u32);

printk("    Reading %d [%x] values from %s, starting at %08x...\n", nCount32, nCount32, ksDriverName, pFPGA);

    int i;
    for (i = 0; i < nCount32; i++)
    {
        *pBuf32++ = *pFPGA++;
    }

    nPos     += nAdjustedCount;
    *pFPos_o  = nPos;

printk(KERN_INFO "%s Read done.  Read %ld (%lx) bytes; nPos=%08lx, FPOS=%08lx \n",
                 ksDriverName, nAdjustedCount, nAdjustedCount, nPos, *pFPos_o);

    return nAdjustedCount;

#undef kFPGA_AlignSize
#undef kFPGA_MinReadSize
}

//*****************************************************************************

static ssize_t
FPGArw_write(struct file     *pFile_io,
                  const char __user *pUserBuf_io,
                  size_t             nCount_i,
                  loff_t            *pFPos_o)
{
#define kFPGA_MinReadSize  (sizeof(u32))
#define kFPGA_AlignSize    (sizeof(u32))+

    //loff_t nPos         = *pFPos_o;
    unsigned long nPos    = *pFPos_o;
    int    nDeviceSize    = FPGA_RegionSize;
    size_t nAdjustedCount = nCount_i;

printk(KERN_INFO "%s WRITE(%p, %x, %08x) \n", ksDriverName, pUserBuf_io, nCount_i, nPos);

    // check limits ...
    if (nPos >= nDeviceSize)               return -EINVAL;
    if ((nPos + nCount_i) > nDeviceSize)   nAdjustedCount = nDeviceSize - nPos;

printk("    Adjusted vals [device] : nPos=%08lx,Count=%08lx\n", nPos, nAdjustedCount);

    if(nAdjustedCount < kFPGA_MinReadSize)
    {
        printk(KERN_WARNING "%s adjusted write count (%d) too small!\n", ksDriverName, nAdjustedCount);
        return -EINVAL;
    }

    if((nPos % kFPGA_AlignSize))
    {
        printk(KERN_WARNING "%s: write misaligned!\n", ksDriverName);
        return -EINVAL;
    }

    u32 __user *pBuf32   = (u32 __user *)pUserBuf_io;
    u32        *pFPGA    = (u32 *)FPGA_pRegs + (nPos/sizeof(u32));
    u32         nCount32 = nAdjustedCount/sizeof(u32);

printk("    Writing %d [%x] values to %s, starting at %08x...\n", nCount32, nCount32, ksDriverName, pFPGA);

    int i;
    for (i = 0; i < nCount32; i++)
    {
        *pFPGA++ = *pBuf32++;
    }

printk("    Done writing %s.\n", ksDriverName);

    nPos     += nAdjustedCount;
    *pFPos_o  = nPos;

    return nAdjustedCount;

#undef kFPGA_AlignSize
#undef kFPGA_MinReadSize
}

//*****************************************************************************

static loff_t
FPGArw_seek (struct file *pFile_io, loff_t nOffs_i, int nWhence_i)
{
#define kFPGA_AlignSize    (sizeof(u32))

    //struct fpga_dev *pDev = pFile_io->private_data;
    loff_t nNewPos;
    int    nDeviceSize = FPGA_RegionSize;

printk("%s SEEK(%d [%08x], %d) \n", ksDriverName, (u32)nOffs_i, (u32)nOffs_i, nWhence_i);
printk("   Current Pos = %08x \n", (u32)pFile_io->f_pos);

    switch(nWhence_i)
    {
        case 0:     // SEEK_SET
            nNewPos = nOffs_i;
        break;

        case 1:     // SEEK_CUR
            nNewPos = pFile_io->f_pos + nOffs_i;
        break;

        case 2:    // SEEK_END
            //nNewPos = pDev->size + nOffs_i;
            nNewPos = nDeviceSize + nOffs_i;
        break;

        default:  // can't happen
        return -EINVAL;
    }

    if (nNewPos < 0)                       return -EINVAL;
    if (nDeviceSize <= nNewPos)            return -EINVAL;
    if (nNewPos % kFPGA_AlignSize)    return -EINVAL;

    pFile_io->f_pos = nNewPos;

printk(KERN_INFO "%s SEEK successful. NewPos = %08x => %08lx \n", ksDriverName, nNewPos, (u32)pFile_io->f_pos);
    return nNewPos;
}

//#############################################################################

static int FPGArw_open (struct inode *inode, struct file *filp)
{
    // allow only one user
    if(test_and_set_bit(0, &FPGA_bIsOpen))
    {
        return -EBUSY;
    }

    FPGA_pRegs = ioremap(FPGA_RegionStart, FPGA_RegionSize);
    if (NULL == FPGA_pRegs)
    {
        printk(KERN_ERR "%s: ioremap failed for %lx : %lx ! \n", ksDriverName, FPGA_pRegs, FPGA_RegionSize);
        return -EIO;
    }

    printk("ioremapped FPGA %lx => %lx, size=%lx\n", FPGA_RegionStart, FPGA_pRegs, FPGA_RegionSize);

    return 0;
}

//*****************************************************************************

static int FPGArw_release(struct inode *inode, struct file *filp)
{
    if (FPGA_pRegs)
    {
        iounmap(FPGA_pRegs);
        FPGA_pRegs = NULL;
    }

    // release the driver for others
    clear_bit(0, &FPGA_bIsOpen);

    return 0;
}

//#############################################################################

/*
 * Set up the cdev structure for a device.
 */
static void FPGA_setup_cdev
(
    struct cdev            *dev,
    int                     minor,
    struct file_operations *fops
)
{
    int err, devno = MKDEV(FPGA_Major, minor);

    cdev_init(dev, fops);
    dev->owner = THIS_MODULE;
    dev->ops   = fops;
    err        = cdev_add (dev, devno, 1);

    /* Fail gracefully if need be */
    if (err)
    {
        printk (KERN_NOTICE "Error %d adding %s%d", err, ksDriverName, minor);
    }
}

//*****************************************************************************

/*
 * Our various sub-devices.
 */

static struct file_operations FPGA_rdwr_ops =
{
    .owner   = THIS_MODULE,
    .open    = FPGArw_open,
    .read    = FPGArw_read,
    .write   = FPGArw_write,
    .llseek  = FPGArw_seek,
    .release = FPGArw_release,
    .mmap    = FPGA_mmap2,
};

//
// TODO: add a minor device for each Architectural Block
//       in the FPGA ..
//

#define MAX_FPGA_DEV   1
#define NUM_FPGA_DEVs (MAX_FPGA_DEV)

/*
 * We export one FPGA devices.  There's no need for us to maintain any
 * special housekeeping info, so we just deal with raw cdev(s).
 */
static struct cdev aoMyDevices[NUM_FPGA_DEVs];

//*****************************************************************************

#if 1
#include "ProcFS.inc.c"
#endif

//#############################################################################

/*
 * Module housekeeping.
 */
static int FPGArw_init(void)
{
    int   result;
    dev_t dev;

    printk("FPGArw_init entered ...");

    // First, setup from any command-line/insmod inputs ...
    //
    FPGA_Major       = major;
    FPGA_Minor       = minor;
    FPGA_RegionStart = start;
    FPGA_RegionSize  = size;
    FPGA_RegionPages = FPGA_RegionSize >> PAGE_SHIFT;
    FPGA_RegionEnd   = FPGA_RegionStart + FPGA_RegionSize - 1;

    dev = MKDEV(FPGA_Major, FPGA_Minor);

    // Figure out our device number.
    if (FPGA_Major)
    {
        result = register_chrdev_region(dev, NUM_FPGA_DEVs, ksDriverName);
    }
    else
    {
        result = alloc_chrdev_region(&dev, 0, NUM_FPGA_DEVs, ksDriverName);
        FPGA_Major = MAJOR(dev);
    }
    if (result < 0)
    {
        printk(KERN_WARNING "%s: unable to get major %d\n", ksDriverName, FPGA_Major);
        return result;
    }

    if (FPGA_Major == 0)        FPGA_Major = result;

    // Now set up the (minor) cdevs.
    FPGA_setup_cdev(aoMyDevices,     FPGA_Minor,    &FPGA_rdwr_ops);
    FPGA_initProcFS();

    return 0;
}

//*****************************************************************************

static void FPGArw_cleanup(void)
{
    if (FPGA_pRegs)
    {
        iounmap(FPGA_pRegs);
        FPGA_pRegs = NULL;
    }

    FPGA_cleanupProcFS();

    // remove each device ...
    cdev_del(aoMyDevices);

    // and then the region ...
    unregister_chrdev_region(MKDEV(FPGA_Major, FPGA_Minor), NUM_FPGA_DEVs);
}

//#############################################################################

module_init(FPGArw_init);
module_exit(FPGArw_cleanup);

//#############################################################################

