/*
 * The source code in this file can be freely used, adapted,
 * and redistributed in source or binary form, so long as an
 * acknowledgment appears in derived source files.  The citation
 * should list that the code comes from the book "Linux Device
 * Drivers" by Alessandro Rubini and Jonathan Corbet, published
 * by O'Reilly & Associates.   No warranty is attached;
 * we cannot take responsibility for errors or fitness for use.
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>   /* printk() */
#include <linux/slab.h>     /* kmalloc() */
#include <linux/fs.h>       /* everything... */
#include <linux/errno.h>    /* error codes */
#include <linux/types.h>    /* size_t */
#include <linux/mm.h>
#include <linux/kdev_t.h>
#include <asm/page.h>
#include <linux/cdev.h>

#include <linux/device.h>

#include <asm/pgtable.h>

//*****************************************************************************

MODULE_LICENSE("Dual BSD/GPL");

//*****************************************************************************

#define ksDriverName "FPGA"

#define kMyMajorNum (251)
#define kMyMinorNum (1)

static int FPGA_Major = kMyMajorNum;
static int FPGA_Minor = kMyMinorNum;

//*****************************************************************************

// Map a 64-KB region starting an unmanged memory 0x0FB00000 ...
//
#define kFPGA_RegionStart    (0x0FB00000)
#define kFPGA_RegionSize     (0x00010000)
#define kFPGA_RegionPages    (kFPGA_RegionSize >> PAGE_SHIFT)
#define kFPGA_RegionEnd      (kFPGA_RegionStart + kFPGA_RegionSize - 1)

static unsigned long FPGA_RegionStart = (kFPGA_RegionStart);
static unsigned long FPGA_RegionSize  = (kFPGA_RegionSize);
static unsigned long FPGA_RegionPages = (kFPGA_RegionPages);
static unsigned long FPGA_RegionEnd   = (kFPGA_RegionEnd);

//*****************************************************************************

static int major = kMyMajorNum;
module_param(major, int, 0);

static int minor = kMyMinorNum;
module_param(minor, int, 0);

static unsigned long start = kFPGA_RegionStart;
module_param(start, ulong, 0);

static unsigned long size = kFPGA_RegionSize;
module_param(size,  ulong, 0);

//*****************************************************************************

/*
 * Open the device; in fact, there's nothing to do here.
 */
static int FPGA_open (struct inode *inode, struct file *filp)
{
    return 0;
}


/*
 * Closing is just as simple.
 */
static int FPGA_release(struct inode *inode, struct file *filp)
{
    return 0;
}

//*****************************************************************************

/*
 * Common VMA ops.
 */

void FPGA_vma_open(struct vm_area_struct *vma)
{
    printk(KERN_NOTICE "%s VMA open, virt %lx, phys %lx\n",
            ksDriverName, vma->vm_start, vma->vm_pgoff << PAGE_SHIFT);
}

void FPGA_vma_close(struct vm_area_struct *vma)
{
    printk(KERN_NOTICE "%s VMA close.\n", ksDriverName);
}

//*****************************************************************************

static struct vm_operations_struct FPGA_remap_vm_ops =
{
    .open =  FPGA_vma_open,
    .close = FPGA_vma_close,
};

//*****************************************************************************

static int FPGA_remap_mmap(struct file *filp, struct vm_area_struct *vma)
{
    // The kernel passes in  the mv_pgoff - the offset, in *PAGES
    //     from the start of the region/buffer that the user app
    //     wants to mmap.
    //     This is the *LAST* argument in the mmap call.
    //     We check it to make sure that it is not OOB.
    //
    if (vma->vm_pgoff > FPGA_RegionPages)    return -EINVAL;

    vma->vm_start = (unsigned long)FPGA_RegionStart;
    vma->vm_end   = (unsigned long)FPGA_RegionEnd;

    // Before remap() is called, pgoff *MUST* be set to the kernel page
    //   offset from zero.  Otherwise, we'll segment fault when we try
    //   to write to the memory (trust me).
    //   Due to this, we figure out the address, (<< PAGE_SHIFT), add it
    //   to the start of the the 'buffer', and turn that back to
    //   pages (>> PAGE_SHIFT).
    //
    vma->vm_pgoff = ((vma->vm_pgoff << PAGE_SHIFT)
                      +  (unsigned long)FPGA_RegionStart)
                      >> PAGE_SHIFT;

    // VM_MAYSHARE is needed for nommu (if we ever do that....heh)
    //
    vma->vm_flags |= (VM_IO | VM_RESERVED | VM_MAYSHARE | VM_SHARED);
    vma->vm_flags |= (VM_IO | VM_RESERVED | VM_READ | VM_WRITE | VM_MAYREAD | VM_MAYWRITE);

    if (remap_pfn_range(vma,
                        vma->vm_start,
                        vma->vm_pgoff,
                        vma->vm_end - vma->vm_start,
                        vma->vm_page_prot))
        return -EAGAIN;

    vma->vm_ops = &FPGA_remap_vm_ops;
    FPGA_vma_open(vma);

    return 0;
}

//*****************************************************************************

ssize_t FPGA_read(struct file *pFile_io,
                       char __user *pBuf_io,
                       size_t       nCount_i,
                       loff_t      *pFPos_o)
{
    return nCount_i;
}

//*****************************************************************************

//*****************************************************************************

/*
 * Set up the cdev structure for a device.
 */
static void FPGA_setup_cdev
(
    struct cdev            *dev,
    int                     minor,
    struct file_operations *fops
)
{
    int err, devno = MKDEV(FPGA_Major, minor);

    cdev_init(dev, fops);
    dev->owner = THIS_MODULE;
    dev->ops   = fops;
    err        = cdev_add (dev, devno, 1);

    /* Fail gracefully if need be */
    if (err)
    {
        printk (KERN_NOTICE "Error %d adding %s%d", err, ksDriverName, minor);
    }
}


/*
 * Our various sub-devices.
 */

static struct file_operations FPGA_remap_ops =
{
    .owner   = THIS_MODULE,
    .open    = FPGA_open,
    .read    = FPGA_read,
    .release = FPGA_release,
    .mmap    = FPGA_remap_mmap,
};


#define MAX_FPGA_DEV   1
#define NUM_FPGA_DEVs (MAX_FPGA_DEV)

/*
 * We export one FPGA devices.  There's no need for us to maintain any
 * special housekeeping info, so we just deal with raw cdev(s).
 */
static struct cdev aoMyDevices[NUM_FPGA_DEVs];

//*****************************************************************************

#include "ProcFS.inc.c"

//*****************************************************************************

/*
 * Module housekeeping.
 */
static int FPGA_init(void)
{
    int   result;
    dev_t dev;

    // First, setup from any command-line/insmod inputs ...
    //
    FPGA_Major       = major;
    FPGA_Minor       = minor;
    FPGA_RegionStart = start;
    FPGA_RegionSize  = size;
    FPGA_RegionPages = FPGA_RegionSize >> PAGE_SHIFT;
    FPGA_RegionEnd   = FPGA_RegionStart + FPGA_RegionSize - 1;

    dev = MKDEV(FPGA_Major, FPGA_Minor);

    /* Figure out our device number. */
    if (FPGA_Major)
    {
        result = register_chrdev_region(dev, NUM_FPGA_DEVs, ksDriverName);
    }
    else
    {
        result = alloc_chrdev_region(&dev, 0, NUM_FPGA_DEVs, ksDriverName);
        FPGA_Major = MAJOR(dev);
    }
    if (result < 0)
    {
        printk(KERN_WARNING "%s: unable to get major %d\n", ksDriverName, FPGA_Major);
        return result;
    }

    if (FPGA_Major == 0)        FPGA_Major = result;

    // Now set up the (minor) cdevs.
    FPGA_setup_cdev(aoMyDevices, FPGA_Minor, &FPGA_remap_ops);
    FPGA_initProcFS();

    return 0;
}


static void FPGA_cleanup(void)
{
    FPGA_cleanupProcFS();

    // remove each device ...
    cdev_del(aoMyDevices);

    // and then the region ...
    unregister_chrdev_region(MKDEV(FPGA_Major, FPGA_Minor), NUM_FPGA_DEVs);
}

//*****************************************************************************

module_init(FPGA_init);
module_exit(FPGA_cleanup);

//*****************************************************************************
