/*
 * banjodecoder.c
 * USL, Inc.
 *
 * This file contains the driver for the Banjo image and audio decoders.
 *
 * Revision 0.1 07/18/2007 dcogley
 * 	- Initial Creation
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/fs.h>
#include <linux/ioport.h>
#include <linux/errno.h>
#include <linux/io.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <linux/delay.h>

#include <asm-ppc/ibm44x.h>
#include <asm/io.h>

#include "banjodecoder.h"

#define DRIVER_VERSION	"0.1"
#define DRIVER_NAME		"banjodecoder"

MODULE_DESCRIPTION("USL, Inc. Image Decoder driver v" DRIVER_VERSION);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Dave Cogley");

int banjo_major = BANJO_MAJOR;
module_param(banjo_major, int, 0);

struct banjo_dev* banjo_device = 0;
void __iomem* banjo_regmap = 0;
void __iomem* gpio1_or = 0;

/* Macros for accessing the indirect EBC registers */
void mtebc(u32 reg, u32 data)
{
	mtdcr(ebccfga, reg);
	mtdcr(ebccfgd, data);
}

u32 mfebc(u32 reg)
{
	mtdcr(ebccfga, reg);
	return mfdcr(ebccfgd);
}

irqreturn_t dma_status_int(int irq, void *dev_id, struct pt_regs *regs)
{
	u32 status;
	u32 pb1ap;
	u32 gpio;

	// get and clear the DMA status
	status = mfdcr(DMA2P40_SR);
	mtdcr(DMA2P40_SR, status);

	// status is TC expire, EOT request (never happens) or error occured
	if (status & DMA0_TC_REACHED ||
		status & DMA0_EOT_REQ ||
		status & DMA0_ERROR)
	{
		if (status & DMA0_ERROR)
		{
			// TODO - error occured
		}

		// drive GPIO48 (EOTn) high to indicate end of transfer 
		gpio = readl(gpio1_or);
		gpio |= EOTN_DRIVE_HIGH;
		writel(gpio, gpio1_or);

		// reset the peripheral ready on EBC perph 1
		pb1ap = mfebc(PB1AP);
		pb1ap |= PB1AP_ENABLE;
		mtebc(PB1AP, pb1ap);
	}

	return IRQ_HANDLED;
}

irqreturn_t frame_req_int(int irq, void *dev_id, struct pt_regs *regs)
{
	u32 gpio;
	u32 p1ap;
	u32 control;

	// set peripheral 1 access permissions to disable "peripheral ready"
	p1ap = mfebc(PB1AP);
	p1ap &= ~PB1AP_ENABLE;
	mtebc(PB1AP, p1ap);

	// drive GPIO48 (EOTn) low 
	gpio = readl(gpio1_or);
	gpio &= ~EOTN_DRIVE_HIGH;
	writel(gpio, gpio1_or);

	// enable the DMA channel
	// the DMA channel is now armed and will begin the transfer immediatly
	// to the configured DMA addresses
	control = mfdcr(DMA2P40_CR0);
	control |= DMA_ENABLE_CHANNEL;
	mtdcr(DMA2P40_CR0, control);

	return IRQ_HANDLED;
}

int banjo_open(struct inode* inode, struct file* filp)
{
	// Find the device and assign to the local file ptr
//	struct banjo_dev* dev = container_of(inode->i_cdev, struct banjo_dev, cdev);
//	filp->private_data = dev;

	return 0;
}

int banjo_release(struct inode* inode, struct file* filp)
{
//	struct banjo_dev* dev = filp->private_data; // device information

	return 0;
}

ssize_t banjo_write(struct file* filp, const char __user* buf, size_t count, loff_t* f_pos)
{
	ssize_t sret = -ENOSYS;
	struct banjo_dev* dev = filp->private_data; // device information
 
	return sret;
}

int banjo_ioctl(struct inode* inode, struct file* filp, unsigned int cmd, unsigned long arg)
{
	int err = 0, ret = 0;
	u32 status;

	/* don't even decode wrong cmds: better returning  ENOTTY than EFAULT */
	if (_IOC_TYPE(cmd) != BANJO_IOC_MAGIC)
		return -ENOTTY;
	if (_IOC_NR(cmd) > BANJO_IOC_MAXNR)
		return -ENOTTY;

	/*
	 * the type is a bitmask, and VERIFY_WRITE catches R/W
	 * transfers. Note that the type is user-oriented, while
	 * verify_area is kernel-oriented, so the concept of "read" and
	 * "write" is reversed
	 */
	if (_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void __user*)arg, _IOC_SIZE(cmd));
	else if (_IOC_DIR(cmd) & _IOC_WRITE)
		err =  !access_ok(VERIFY_READ, (void __user*)arg, _IOC_SIZE(cmd));
	if (err)
		return -EFAULT;

	switch (cmd)
	{
		// reset the banjo decoder
		case BANJO_IOCDECODERRESET:
			writel(0xffffffff, banjo_regmap + BANJO_RESET_ADDR);
			mdelay(BANJO_RESET_DELAY);
			writel(0x00000000, banjo_regmap + BANJO_RESET_ADDR);
		break;
		case BANJO_IOCDECODERSTATUS:
			status = readl(banjo_regmap + BANJO_STATUS_ADDR);
			ret = __put_user(status, (int __user*)arg);
		break;
	}

	return ret;
}

int banjo_dmastatus_proc(char* buf, char** start, off_t offset, int count, int* eof, void* data)
{
	int len = 0;

	len += sprintf(buf + len, "\nBanjo Decoder - DMA");
	len += sprintf(buf + len, "\n\tDCRN_DMACR0\t: %08X", mfdcr(DMA2P40_CR0));
	len += sprintf(buf + len, "\n\tDMA2P40_CTC0\t: %08X", mfdcr(DMA2P40_CTC0));
	len += sprintf(buf + len, "\n\tDCRN_DMASR\t: %08X", mfdcr(DMA2P40_SR));
	len += sprintf(buf + len, "\n\n");

	*eof = 1;
	return len;
}

struct file_operations banjo_fops = 
{
	.owner =     THIS_MODULE,
	.llseek =    0,
	.read =	     0,	// status read through iocontrol
	.write =     banjo_write,
	.ioctl =     banjo_ioctl,
	.open =	     banjo_open,
	.release =   banjo_release,
	.aio_read =  0,
	.aio_write = 0,
};

static void banjo_setup_cdev(struct banjo_dev* dev, int index)
{
	int devno = MKDEV(banjo_major, index);
    
	cdev_init(&dev->cdev, &banjo_fops);

	dev->cdev.owner = THIS_MODULE;
	dev->cdev.ops = &banjo_fops;
	cdev_add(&dev->cdev, devno, 1);
}

static void set_dma_buffers(int n)
{
	u32 addr;

	// setup count and control register
	mtdcr(DMA2P40_CTC0, DMA_CONTROLTC0);

	// 64b source address
	mtdcr(DMA2P40_SAH0, 0x00000000);	// must be zero
	switch (n)
	{
		case 0:
			addr = DMABUF0;
		break;
		case 1:
			addr = DMABUF1;
		break;
		case 2:
			addr = DMABUF2;
		break;
		case 3:
			addr = DMABUF3;
		break;
	}
	mtdcr(DMA2P40_SAL0, addr);
	
	// destination address is fixed to the banjo decoder media address
	// the destination address really does not matter to the FPGA but
	// because of address alignment problems we want to start at the
	// base address
	mtdcr(DMA2P40_DAH0, 0x00000001);	// destination is addressable on the EBC
	mtdcr(DMA2P40_DAL0, BANJO_DECODER_BASE);
}

static void __init dma_init(void)
{
	// setup the channel 0 control register
	mtdcr(DMA2P40_CR0, DMA_CONFIG0);

	// setup to transfer from the first DMA buffer
	set_dma_buffers(0);
}

static int __init banjo_init(void)
{
	int result;
	dev_t dev = MKDEV(banjo_major, 0);

	printk(KERN_INFO "USL, Inc. Image Decoder driver v" DRIVER_VERSION "\n");

	/*
	 * Register your major, and accept a dynamic number.
	 */
	if (banjo_major)
		result = register_chrdev_region(dev, 0, DRIVER_NAME);
	else 
	{
		result = alloc_chrdev_region(&dev, 0, 0, DRIVER_NAME);
		banjo_major = MAJOR(dev);
	}
	if (result < 0)
		return result;

	// remap the FPGA registers
	request_mem_region(BANJO_DECODER_BASE, BANJO_REGISTERMAP_SIZE + 1, DRIVER_NAME);
	banjo_regmap = ioremap(BANJO_DECODER_BASE, BANJO_REGISTERMAP_SIZE);

	// remap GPIO1 port address
	request_mem_region(GPIO1_OR, sizeof (long) + 1, DRIVER_NAME);
	gpio1_or = ioremap(GPIO1_OR, sizeof (long));
	
	// initialize DMA
	dma_init();
	
	// allocate all device structures
	banjo_device = kmalloc(sizeof (struct banjo_dev), GFP_KERNEL);
	memset(banjo_device, 0, sizeof (struct banjo_dev));

	// register character devices with the kernel
	banjo_setup_cdev(banjo_device, 0);

	// create /proc fs entries
	create_proc_read_entry("banjodma", 0, NULL, banjo_dmastatus_proc, NULL);

	// map interrupt handler for DMA status
	request_irq(DMA_STATUS_IRQ, dma_status_int, SA_INTERRUPT, 
		DRIVER_NAME, dma_status_int);	
	
	// map interrupt handler for frame ready interrupt IRQ7
	request_irq(FRAME_READY_IRQ, frame_req_int, SA_INTERRUPT, 
		DRIVER_NAME, frame_req_int);	

	return 0;
}

static void __exit banjo_exit(void)
{
	free_irq(FRAME_READY_IRQ, NULL);
	free_irq(DMA_STATUS_IRQ, NULL);

	cdev_del(&banjo_device->cdev);
	kfree(banjo_device);

	remove_proc_entry("banjodma", NULL);

	iounmap(gpio1_or);
	iounmap(banjo_regmap);

	release_mem_region(BANJO_DECODER_BASE, BANJO_REGISTERMAP_SIZE + 1);
	release_mem_region(GPIO1_OR, sizeof (long) + 1);

	unregister_chrdev_region(MKDEV(banjo_major, 0), 1);
}

module_init(banjo_init);
module_exit(banjo_exit);
