/*
 * sig_fork.c: race a killing signal against a fork() system call.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright (C) IBM Corporation, 2005
 *
 * Author: Paul E. McKenney <paulmck@us.ibm.com>
 */

#include "steamroller.h"

struct sig_test_ctrl {
	int startflag;
	int raced;
	int childforkdone;
	int grandchildrunning;
};

int test_sig_dfl__fork(void *p, int parentspin, long childcpuset)
{
	int childpid;
	int i;
	int parentpid;
	int pid;
	int status;
	struct sig_test_ctrl *stp = (struct sig_test_ctrl *)p;

	stp->startflag = 0;
	stp->raced = 0;
	stp->childforkdone = 0;
	stp->grandchildrunning = 0;
	parentpid = getpid();
	if ((pid = fork()) == 0) {
		childpid = getpid();
		sched_setaffinity(0, sizeof(childcpuset), &childcpuset);
		stp->startflag = 1;
		spindelay(us2spindelay(100));
		stp->raced = 1;
		if ((pid = fork()) == 0) {
			stp->grandchildrunning = 1;
		} else if (pid > 0) {
			stp->childforkdone = 1;
		} else {
			perror("fork grandchild");
			if (getpid() != parentpid) {
				fprintf(stderr,
					"Error fork(): child: %d mine(gc):%d\n",
					childpid, getpid());
				exit(-1);
			}
		}
		exit(0);
	}
	if (pid == -1) {
		perror("fork");
		if (getpid() != parentpid) {
			fprintf(stderr,
				"Error fork() child: parent: %d mine:%d\n",
				parentpid, getpid());
			exit(-1);
		}
		exit(-1);
	}
	while (stp->startflag == 0) continue;
	spindelay(parentspin);
	if (waitpid(pid, &status, WNOHANG) != 0) {
		return STEAMROLLER_LATE;
	} else {
		if (kill(pid, SIGINT) != 0) {
			perror("kill");
			exit(-1);
		}
		wait(&status);
		if (!stp->raced) {
			return STEAMROLLER_EARLY;
		}
		if (!stp->childforkdone && !stp->grandchildrunning) {
			return STEAMROLLER_RACED;
		} else {
			return STEAMROLLER_LATE;
		}
	}
}

int main(int argc, char *argv[])
{
	long childcpuset;
	void *p;

	childcpuset = steamroller_init(argc, argv);
	p = (int *)mapmem(sizeof(struct sig_test_ctrl), -1);
	search_and_steamroller(test_sig_dfl__fork, p, childcpuset);
}
