/***************************************************************************
 *                                                                         *
 *  Simple XML Parser.                                                     *
 *     copyright (C)  Tim Hockin <thockin@hockin.org>                      *
 *                                                                         *
 *  Heavily modified by                                                    *
 *     copyright (C) 2004, 2005  Michael Buesch <mbuesch@freenet.de>       *
 *                                                                         *
 * Released under the LGPL version 2.1 (http://www.gnu.org)                *
 * This program is free software; you can redistribute it and/or modify    *
 * it under the terms of the GNU Lesser General Public License version 2.1 *
 * as published by the Free Software Foundation.                           *
 *                                                                         *
 ***************************************************************************/

#include "xmlparser.h"
#include "xmlutil.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#define XML_PAGE "<fieldbus name=Ahmad length=12><board  type=INTEL>Asus</board><board><board device name>\
	                /dev/pbboard0</board device name></board></fieldbus>"


#define STRING_CAP_STEP		100 /* byte */


struct xml_context
{
	const char *buf;
	size_t size;
	size_t pos;
	unsigned int lineno;
};

typedef enum xml_state
{
	ST_TAGSTART,
	ST_START_STAG,
	ST_STAG_NAME_DONE,
	ST_PROPNAME,
	ST_GETEQ,
	ST_GOTEQ,
	ST_PROPVAL,
	ST_SQ_PROPVAL,
	ST_POST_PROPVAL,
	ST_END_SETAG,
	ST_END_STAG,
	ST_CONTENT,
	ST_MAYBE_COMMENT,
	ST_NEWTAG_START,
	ST_START_ETAG,
	ST_ETAG_NAME,
	ST_END_ETAG
} state_t;


struct xml_element* get_first_child(struct xml_element *parent){
	
	if(parent->first_child)
		return parent->first_child;	
	return NULL;
}

struct xml_element* get_next_element(struct xml_element *element, struct xml_element *child){
	
	if(child)
		return child->next;
		
	return NULL;
	
}

struct xml_element* get_next_node(struct xml_element *element){
	
	if(element->next)
		return element->next;	
	return NULL;
	
}

char * get_element_name(struct xml_element *element){

	 if(element->name)
		return element->name->s;
	 return 0;
}

char * get_element_content(struct xml_element *element){

	if(element->content)	
		return element->content->s;
	return 0;
}
struct xml_prop *get_next_attribute( struct xml_element *element, struct xml_prop* attribute){
	if(!element->first_prop)
		return NULL;
	if(attribute){
		return attribute->next;}
	else
		return element->first_prop;
}

char *get_attribute_name(struct xml_prop* attribute){
	if(attribute->name->s)
	return attribute->name->s;
	return NULL;
}


char *get_attribute_value(struct xml_prop* attribute){
	return attribute->value->s;
}

struct xml_element *get_element_parent(struct xml_element *element){
	if(element->parent)
	return element->parent;
	else return NULL;
}




static struct xml_string * alloc_xml_string(void)
{
	struct xml_string *s;

	s = malloc(sizeof(*s));
	if (!s) {
		printf("Out of memory\n");
		return 0;
	}
	s->s = malloc(STRING_CAP_STEP);
	if (!s->s) {
		printf("Out of memory\n");
		free(s);
		return 0;
	}
	s->s[0] = '\0';
	s->size = 0;
	s->_capacity = STRING_CAP_STEP;

	return s;
}

static void free_xml_string(struct xml_string *s)
{
	if (!s)
		return;
	free(s->s);
	free(s);
}

static int xml_string_append_c(struct xml_string *s, char c)
{
	if (s->size + 1 == s->_capacity) {
		size_t new_cap;
		char *new_buf;

		new_cap = s->_capacity + STRING_CAP_STEP;
		new_buf = realloc(s->s, new_cap);
		if (!new_buf) {
			printf("Out of memory\n");
			return 1;
		}
		s->s = new_buf;
		s->_capacity = new_cap;
	}
	s->size++;
	s->s[s->size - 1] = c;
	s->s[s->size] = '\0';
	return 0;
}

static int xml_string_copy(struct xml_string *to, struct xml_string *from)
{
	if (from->size + 1 > to->_capacity) {
		char *new_buf;

		new_buf = realloc(to->s, from->size + 1);
		if (!new_buf) {
			printf("Out of memory\n");
			return 1;
		}
		to->s = new_buf;
		to->_capacity = from->size + 1;
	}
	memcpy(to->s, from->s, from->size + 1);
	to->size = from->size;
	return 0;
}

static void xml_string_clear(struct xml_string *s)
{
	s->size = 0;
	/* We don't reset capacity here. */
}

static struct xml_prop * alloc_xml_prop(void)
{
	struct xml_prop *p;

	p = malloc(sizeof(*p));
	if (!p) {
		printf("Out of memory\n");
		return 0;
	}
	p->name = alloc_xml_string();
	if (!p->name) {
		free(p);
		return 0;
	}
	p->value = alloc_xml_string();
	if (!p->value) {
		free_xml_string(p->name);
		free(p);
		return 0;
	}
	p->next = 0;
	p->prev = 0;

	return p;
}

static void free_xml_prop(struct xml_prop *p)
{
	if (!p)
		return;
	free_xml_string(p->value);
	free_xml_string(p->name);
	free(p);
}

static struct xml_element * alloc_xml_element(void)
{
	struct xml_element *e;

	e = malloc(sizeof(*e));
	if (!e) {
		printf("Out of memory\n");
		return 0;
	}
	e->name = alloc_xml_string();
	if (!e->name) {
		free(e);
		return 0;
	}
	e->content = alloc_xml_string();
	if (!e->content) {
		free_xml_string(e->name);
		free(e);
		return 0;
	}
	e->first_prop = 0;
	e->last_prop = 0;
	e->first_child = 0;
	e->last_child = 0;
	e->next = 0;
	e->prev = 0;
	e->parent = 0;
	return e;
}

void xml_free_element(struct xml_element *e)
{
	struct xml_element *child;
	struct xml_prop *prop;

	if (!e)
		return;

	child = e->first_child;
	while (child) {
		struct xml_element *del;

		del = child;
		child = child->next;
		xml_free_element(del);
	}

	prop = e->first_prop;
	while (prop) {
		struct xml_prop *del;

		del = prop;
		prop = prop->next;
		free_xml_prop(del);
	}

	free_xml_string(e->name);
	free_xml_string(e->content);
	free(e);
}

static void xml_element_append_child(struct xml_element *parent,
				     struct xml_element *child)
{
	child->next = 0;
	child->prev = 0;
	if (!parent->first_child)
		parent->first_child = child;
	if (parent->last_child) {
		parent->last_child->next = child;
		child->prev = parent->last_child;
	}
	parent->last_child = child;
	child->parent=parent;
}

static void xml_element_append_prop(struct xml_element *e,
				    struct xml_prop *prop)
{
	prop->next = 0;
	prop->prev = 0;
	if (!e->first_prop)
		e->first_prop = prop;
	if (e->last_prop) {
		e->last_prop->next = prop;
		prop->prev = e->last_prop;
	}
	e->last_prop = prop;
}

static int is_name_start(char c)
{
	return (simple_isalpha(c) || c == '_' || c == ':');
}

static int is_name(char c)
{
	return (is_name_start(c) ||
		simple_isdigit(c) ||
		c == '.' ||
		c == '-');
}

static int next_char(struct xml_context *ctx, char *c)
{
	if (ctx->pos == ctx->size){
		return 0;}
	*c = ctx->buf[ctx->pos];

	if (*c == '\0'){
		return 0;}
	if (*c == '\n'){
		ctx->lineno++;}
	ctx->pos++;
	return 1;
}

static void prolog(struct xml_context *ctx)
{
	char cur = ' ';
	char prev = ' ';

	while (next_char(ctx, &cur)) {
		if (cur == '>' && prev == '?')
			return;
		prev = cur;
	}
}

static void comment(struct xml_context *ctx)
{
	char cur = ' ';
	char prev = ' ';
	char prev2 = ' ';

	while (next_char(ctx, &cur)) {
		if (cur == '>' && prev == '-' && prev2 == '-')
			return;
		prev2 = prev;
		prev = cur;
	}
}

static char unescape(struct xml_context *ctx)
{
	char buf[16];
	unsigned int i = 0;
	const char *escapes[] = {
		"lt;",
		"gt;",
		"amp;",
		"apos;",
		"quot;",
		"nbsp;",
		NULL
	};
	const char esc_vals[] = {
		'<',
		'>',
		'&',
		'\'',
		'"',
		' ',
		'\0'
	};

	/* we assume we got a '&' before we got here */
	if (!next_char(ctx, buf + i)) {
		printf("ERROR: premature EOF (line %d)\n", ctx->lineno);
		return -1;
	}
	while ((buf[i] != ';') && (!simple_isspace(buf[i])) && (i < sizeof(buf)-1)) {
		i++;
		if (!next_char(ctx, buf + i)) {
			printf("ERROR: premature EOF (line %d)\n", ctx->lineno);
			return -1;
		}
	}
	buf[i + 1] = '\0';

	/* got an escape */
	if (buf[i] == ';') {
		i = 0;
		/* got a &# style escape */
		if (buf[0] == '#') {
			char charesc = -1;
			int base = 10;
			i++;
			if (simple_tolower(buf[i]) == 'x') {
				base = 16;
				i++;
			}
			while (buf[i] != ';') {
				int decval;

				/* convert buf[i]) */
				if (simple_isdigit(buf[i])) {
					decval = buf[i] - '0';
				} else if (simple_isxdigit(buf[i])) {
					decval = (simple_tolower(buf[i]) - 'a') + 10;
				} else {
					return -1;
				}
				charesc *= base;
				charesc += decval;
			}
			return charesc;
		}
		/* got a 'normal' escape */
		while (escapes[i]) {
			if (!simple_strcasecmp(buf, escapes[i])) {
				return esc_vals[i];
			}
			i++;
		}
	}

	return -1;
}

static int add_content_char(struct xml_context *ctx,
			    struct xml_element *e,
			    char c)
{
	if (e->content->size == 0) {
		/* This is the first phase of the content
		 * string whitespace cleanup.
		 * Second phase is in cleanup_string()
		 */
		if (simple_isspace(c) || c == '\n'){

			return 0;}
	}
	return xml_string_append_c(e->content, c);
}

static void cleanup_string(struct xml_string *s)
{
	size_t i;
	char c;

	/* This is the second phase of the content
	 * string whitespace cleanup.
	 * The first phase is in add_content_char()
	 */
	if (s->size <= 0)
		return;
	i = s->size - 1;
	do {
		c = s->s[i];
	} while ((simple_isspace(c) || c == '\n') && (i-- > 0));
	s->size = i + 1;
	s->s[s->size] = '\0';
}

struct xml_element * parse_element(struct xml_context *ctx, char first)
{
	char c;
	state_t state = ST_TAGSTART;
	struct xml_element *e = 0;
	struct xml_prop *p = 0;
	struct xml_element *ret = 0;
	int flag=0;

	struct xml_string *propname = 0;
	struct xml_string *propval = 0;
	struct xml_string *endname = 0;

	ret = alloc_xml_element();
	if (!ret)
		goto out_fail;
	propname = alloc_xml_string();
	if (!propname)
		goto out_fail;
	propval = alloc_xml_string();
	if (!propval)
		goto out_fail;
	endname = alloc_xml_string();
	if (!endname)
		goto out_fail;

	/* we got a '<' */
	if (is_name_start(first)) {
		if (xml_string_append_c(ret->name, simple_tolower(first)))
			goto out_fail;
		state = ST_START_STAG;
	} else
		state = ST_TAGSTART;

	while (next_char(ctx, &c)) {
		switch (state) {
			/* start */
			case ST_TAGSTART:
				if (simple_isspace(c)) {
					/* state = CURRENT_STATE; */
				} else if (is_name_start(c)) {
					if (xml_string_append_c(ret->name, simple_tolower(c)))
						goto out_fail;
					state = ST_START_STAG;
				} else
					goto out_inval_char;
				break;
			/* we got the start of a keyword for an stag */
			case ST_START_STAG:
				if (is_name(c)) {
					if (xml_string_append_c(ret->name, simple_tolower(c)))
						goto out_fail;
					/* state = CURRENT_STATE; */
				} else if (simple_isspace(c))
					state = ST_STAG_NAME_DONE;
				else if (c == '>')
					state = ST_END_STAG;
				else if (c == '/')
					state = ST_END_SETAG;
				else
					goto out_inval_char;
				break;
			/* keyword done - look for next prop or other */
			case ST_STAG_NAME_DONE:
				if (simple_isspace(c)) {
					/* state = CURRENT_STATE; */
				} else if (c == '>'){
					state = ST_END_STAG;}
				else if (c == '/')
					state = ST_END_SETAG;
				else if (is_name_start(c)) {
					/* must be a propname */
					if (xml_string_append_c(propname, simple_tolower(c)))
						goto out_fail;
					state = ST_PROPNAME;
				} else
					goto out_inval_char;
				break;
			/* found start of a propname */
			case ST_PROPNAME:
				if (is_name(c)) {
					if (xml_string_append_c(propname, simple_tolower(c)))
						goto out_fail;
					/* state = CURRENT_STATE; */
				} else if (c == '=')
					state = ST_GOTEQ;
				else if (simple_isspace(c))
					state = ST_GETEQ;
				else
					goto out_inval_char;
				break;
			/* got propname, get '=' */
			case ST_GETEQ:
				if (simple_isspace(c)) {
					/* state = CURRENT_STATE; */
				} else if (c == '=')
					state = ST_GOTEQ;
				else
					goto out_inval_char;
				break;
			/* got '=' - find propval */
			case ST_GOTEQ:
				if (simple_isspace(c)) {
					/* state = CURRENT_STATE; */
				} else if (c == '"')
					state = ST_PROPVAL;
				else if (c == '\'')
					state = ST_SQ_PROPVAL;
				else
					goto out_inval_char;
				break;
			/* got a quoted propval */
			case ST_PROPVAL:
				if (c == '&') {
					char lc = unescape(ctx);
					if (lc > 0) {
						if (xml_string_append_c(propval, lc))
							goto out_fail;
					}
				} else if (c == '"') {
					p = alloc_xml_prop();
					if (!p)
						goto out_fail;
					if (xml_string_copy(p->name, propname))
						goto out_fail;
					if (xml_string_copy(p->value, propval))
						goto out_fail;
					xml_element_append_prop(ret, p);
					p = 0;
					xml_string_clear(propname);
					xml_string_clear(propval);
					state = ST_STAG_NAME_DONE;
				} else if (c == '<')
					goto out_inval_char;
				else {
					if (xml_string_append_c(propval, c))
						goto out_fail;
					/* state = CURRENT_STATE; */
				}
				break;
			/* got a single-quoted propval */
			case ST_SQ_PROPVAL:
				if (c == '&') {
					char lc = unescape(ctx);
					if (lc > 0) {
						if (xml_string_append_c(propval, lc))
							goto out_fail;
					}
				} else if (c == '\'') {
					p = alloc_xml_prop();
					if (!p)
						goto out_fail;
					if (xml_string_copy(p->name, propname))
						goto out_fail;
					if (xml_string_copy(p->value, propval))
						goto out_fail;
					xml_element_append_prop(ret, p);
					p = 0;
					xml_string_clear(propname);
					xml_string_clear(propval);
					state = ST_STAG_NAME_DONE;
				} else if (c == '<') {
					/* spec says this is an error */
					goto out_inval_char;
				} else {
					if (xml_string_append_c(propval, c))
						goto out_fail;
					/* state = CURRENT_STATE; */
				}
				break;
			/* spec says we must have a space between props */
			case ST_POST_PROPVAL:
				if (simple_isspace(c))
					state = ST_STAG_NAME_DONE;
				else if (c == '>')
					state = ST_END_STAG;
				else if (c == '/')
					state = ST_END_SETAG;
				else
					goto out_inval_char;
				break;
			/* got a '/', suggesting an empty element */
			case ST_END_SETAG:
				if (simple_isspace(c)) {
					/* state = CURRENT_STATE; */
				} else if (c == '>')
					goto out_success;
				else
					goto out_inval_char;
				break;
			/* time to look for content */
			case ST_END_STAG:
				if (simple_isspace(c)) {
					/* state = CURRENT_STATE; */
				} else if (c == '&') {
					char lc = unescape(ctx);
					if (lc > 0) {
						if (add_content_char(ctx, ret, lc))
							goto out_fail;
					}
					state = ST_CONTENT;
				} else if (c == '<'){
					state = ST_MAYBE_COMMENT;}
				else {
					if (add_content_char(ctx, ret, c))
						goto out_fail;
					state = ST_CONTENT;
				}
				break;
			/* processing content */
			case ST_CONTENT:
				if (simple_isspace(c)) {
					if (add_content_char(ctx, ret, c)){		
						printf("element %s\n", get_element_name(ret) );
						printf("fail2\n");
						goto out_fail;}
					/* state = CURRENT_STATE; */
				}		   
				else if (c == '<')
					state = ST_MAYBE_COMMENT;
				else if (c == '&') {
					char lc = unescape(ctx);
					if (lc > 0) {
						if (add_content_char(ctx, ret, lc))
									goto out_fail;
							}
				}																      
				else {
					if (add_content_char(ctx, ret, c)){
						goto out_fail;}
					/* state = CURRENT_STATE; */
				}
				break;
			/* got a '<' - check for a comment */
			case ST_MAYBE_COMMENT:
				if (c == '!')
					comment(ctx);
				else if (simple_isspace(c))
					state = ST_NEWTAG_START;
				else if (c == '/')
					state = ST_START_ETAG;
				else {
					e = parse_element(ctx, c);
					if (!e)
						goto out_fail;
					xml_element_append_child(ret, e);
					e = 0;
					state = ST_CONTENT;
					flag=1;
				}
				break;
			/* got a '<' in content - figure it out */
			case ST_NEWTAG_START:
				if (simple_isspace(c)) {
					/* state = CURRENT_STATE; */
				} else if (c == '/')
					state = ST_START_ETAG;
				else {
					e = parse_element(ctx, c);
					if (!e)
						goto out_fail;
					xml_element_append_child(ret, e);
					e = 0;
					state = ST_CONTENT;
				}
				break;
			/* got a </ */
			case ST_START_ETAG:
				if (simple_isspace(c)) {
					/* state = CURRENT_STATE; */
				} else if (is_name_start(c)) {
					if (xml_string_append_c(endname, c))
						goto out_fail;
					state = ST_ETAG_NAME;
				} else
					goto out_inval_char;
				break;
			/* reading all of </name */
			case ST_ETAG_NAME:
				if (is_name(c)) {
					if (xml_string_append_c(endname, c))
						goto out_fail;
					/* state = CURRENT_STATE; */
				} else if (simple_isspace(c))
					state = ST_END_ETAG;
				else if (c == '>') {
					if (ret->name->size == endname->size &&
					    simple_strncasecmp(ret->name->s, endname->s, endname->size) == 0) {
						goto out_success;
					} else {
						/* not a match! */
						printf("endname (%s) != element->name (%s)\n",
						       endname->s, ret->name->s);
						state = ST_CONTENT;
						xml_string_clear(endname);
					}
				} else
					goto out_inval_char;
				break;
			/* just get '>' and wrap up */
			case ST_END_ETAG:
				if (simple_isspace(c)) {
					/* state = CURRENT_STATE; */
				} else if (c == '>') {
					if (ret->name->size == endname->size &&
					    simple_strncasecmp(ret->name->s, endname->s, endname->size) == 0) {
						goto out_success;
					} else {
						/* not a match! */
						printf("endname (%s) != element->name (%s)\n",
						       endname->s, ret->name->s);
						state = ST_CONTENT;
						xml_string_clear(endname);
					}
				} else
					goto out_inval_char;
				break;
		}
	}
	printf("ERROR: premature EOF (line %d)\n", ctx->lineno);
	goto out_fail;

out_inval_char:
	printf("invalid char '%c' in state %d (line %d)\n",
	       c, state, ctx->lineno);
out_fail:
	xml_free_element(ret);
	xml_free_element(e);
	free_xml_prop(p);
	free_xml_string(propname);
	free_xml_string(propval);
	free_xml_string(endname);
	return 0;

out_success:
	free_xml_string(propname);
	free_xml_string(propval);
	free_xml_string(endname);
	cleanup_string(ret->content);
	return ret;
}

int xml_parse_buffer(struct xml_buffer buffer, struct xml_element **ret)
{
	struct xml_element *root;
	struct xml_element *e;
	struct xml_context ctx;
	char c;
	int can_prolog = 1;

	root = alloc_xml_element();
	strcpy(root->name->s, "root");
	if (!root)
		return 1;
	ctx.buf = buffer.text;
	ctx.size = buffer.text_size;
	ctx.pos = 0;
	ctx.lineno = 1;

	while (next_char(&ctx, &c)) {
		if (simple_isspace(c))
		continue;
		else if (c == '<') {
			if (!next_char(&ctx, &c))
				goto out_premat_eof;
			/* handle a prolog ("<?") */
			if (c == '?' && can_prolog) {
				prolog(&ctx);
				continue;
			}
			can_prolog = 0;

			/* handle a comment ("<!") */
			if (c == '!') {
				comment(&ctx);
				continue;
			}

			e = parse_element(&ctx, c);
			if (!e)
				goto out_fail;
			xml_element_append_child(root, e);
		} else
			goto out_inval_char;
	}
	//printf("bla");
	*ret = root;
	return 0;

out_premat_eof:
	printf("ERROR: premature EOF (line %d)\n", ctx.lineno);
	goto out_fail;

out_inval_char:
	printf("invalid char '%c' in state top (line %d)\n",
	       c, ctx.lineno);

out_fail:
	printf("out_fail\n");
	xml_free_element(root);
	return 1;
} 

struct xml_buffer  load_xml_file(char *source){
	
	FILE *p_InText;
	struct xml_buffer buffer;
	
    	p_InText = fopen (source , "r");

	if (p_InText == NULL)
	{
		 printf("Fehler beim lesen der Datei \n");
		 exit(1);
	} /*End If */

	else
	{
		 fseek( p_InText, 0 , SEEK_END);
		 buffer.text_size = ftell(p_InText);
		 buffer.text =  (char*) malloc(sizeof(char)*buffer.text_size);
		 rewind(p_InText);
		 fread(buffer.text, 1, buffer.text_size, p_InText);
		return buffer;
	 } /* End else */
	
}


