/* 
 * High resolution timer test software
 *
 * (C) 2005 Thomas Gleixner <tglx@linutronix.de>
 *
 * Inspired by jittertest 
 * Copyright (C) 2004 by Stefan Panwinkler  stefan@PASTLINUX1
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License Veriosn
 * 2 as published by the Free Software Foundation;
 *
 * Depends on HRT support library available from
 * http://sourceforge.net/projects/high-res-timers/
 *
 */

#define VERSION_STRING "V 0.1"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <signal.h>
#include <fcntl.h>
#include <getopt.h>
#include <linux/unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <posix_time.h>

/* Ugly, but .... */
#define gettid() syscall(__NR_gettid)
#define sigev_notify_thread_id _sigev_un._tid

#define USEC_PER_SEC	1000000
#define NSEC_PER_SEC	1000000000

#define MODE_CYCLIC	0
#define MODE_NANOSLEEP	1

/* Must be power of 2 ! */
#define VALBUF_SIZE	16384

/* Struct to transfer parameters to the thread */
struct thread_param {
	int prio;
	int mode;
	int signal;
	int clock;
	unsigned long max_cycles;
	struct thread_stat *stats;
	int bufmsk;
	unsigned long interval;
};

/* Struct for statistics */
struct thread_stat {
	unsigned long cycles;
	unsigned long cyclesread;
	long min;
	long max;
	long act;
	long *values;
	pthread_t thread;
	int threadstarted;
	int tid;
};

static int shutdown;

static inline void tsnorm(struct timespec *ts)
{
	while (ts->tv_nsec > NSEC_PER_SEC) {
		ts->tv_nsec -= NSEC_PER_SEC;
		ts->tv_sec++;
	}
}

static inline long calcdiff(struct timespec t1, struct timespec t2)
{
	long diff;
	diff = USEC_PER_SEC * ((int) t1.tv_sec - (int) t2.tv_sec);
	diff += ((int) t1.tv_nsec - (int) t2.tv_nsec) / 1000;
	return diff;
}

/* 
 * timer thread
 * 
 * Modes:
 * - clock_nanosleep based
 * - cyclic timer based
 * 
 * Clock:
 * - CLOCK_MONOTONIC
 * - CLOCK_REALTIME
 * - CLOCK_MONOTONIC_HR
 * - CLOCK_REALTIME_HR
 * 
 */
void *timerthread(void *param)
{
	struct thread_param *par = param;
	struct sched_param schedp;
	struct sigevent sigev;
	sigset_t sigset;
	timer_t timer;
	struct timespec now, next, interval;
	struct itimerspec tspec;
	struct thread_stat *stat = par->stats;
	int policy = par->prio ? SCHED_FIFO : SCHED_OTHER;	

	interval.tv_sec = 0;
	interval.tv_nsec = par->interval * 1000;
	tsnorm(&interval);

	stat->tid = gettid();

	if (par->mode == MODE_CYCLIC) {

		sigemptyset(&sigset);
		sigaddset(&sigset, par->signal);
		sigprocmask(SIG_BLOCK, &sigset, NULL);

		sigev.sigev_notify = SIGEV_THREAD_ID | SIGEV_SIGNAL;
		sigev.sigev_signo = par->signal;
		sigev.sigev_notify_thread_id = stat->tid;
		timer_create(par->clock, &sigev, &timer);
		tspec.it_interval = interval;
	}

	memset(&schedp, 0, sizeof(schedp));
	schedp.sched_priority = par->prio;
	sched_setscheduler(0, policy, &schedp);

	/* Get current time */
	clock_gettime(par->clock, &now);
	next = now;
	next.tv_sec++;
	if (par->mode == MODE_CYCLIC) {
		tspec.it_value = next;
		timer_settime(timer, TIMER_ABSTIME, &tspec, NULL);
	}
	
	while (!shutdown) {

		long diff;

		/* Wait for next period */
		if (par->mode == MODE_CYCLIC) {
			int sigs;
			if (sigwait(&sigset, &sigs) < 0)
				break;
		} else {
			clock_nanosleep(par->clock, TIMER_ABSTIME, &next, NULL);
		}
		clock_gettime(par->clock, &now);

		diff = calcdiff(now, next);
		if (diff < stat->min)
			stat->min = diff;
		if (diff > stat->max)
			stat->max = diff;
		stat->act = diff;
		stat->cycles++;
		
		if (par->bufmsk)
			stat->values[stat->cycles & par->bufmsk] = diff;

		next.tv_nsec += interval.tv_nsec;
		tsnorm(&next);
		
		if (par->max_cycles && par->max_cycles == stat->cycles)
			break;
	}

	if (par->mode == MODE_CYCLIC)
		timer_delete(timer);

	/* switch to normal */
	schedp.sched_priority = 0;
	sched_setscheduler(0, SCHED_OTHER, &schedp);	

	return NULL;
}


/* Print usage information */
static void display_help(void)
{
	printf("jittertest %s\n", VERSION_STRING);
	printf("Usage:\n"
	       "jittertest <options>\n\n"
	       "-c CLOCK --clock=CLOCK     select clock\n"
	       "                           0 = CLOCK_MONOTONIC\n"
	       "                           1 = CLOCK_REALTIME\n"
	       "                           2 = CLOCK_MONOTONIC_HR (default)\n"
	       "                           3 = CLOCK_REALTIME_HR\n"
	       "-d DIST  --distance=DIST   distance of thread intervals in us default=500\n"
	       "-i INTV  --interval=INTV   base interval of thread in us default=1000\n"
	       "-l LOOPS --loops=LOOPS     number of loops: default=0(endless)\n"
	       "-n       --nanosleep       use clock_nanosleep\n"
	       "-p PRIO  --prio=PRIO       priority of highest prio thread\n"
	       "-t NUM   --threads=NUM     number of threads: default=1\n"
	       "-v       --verbose         output values on stdout for statistic\n"
	       "                           format: n:c:v n=tasknum c=count v=value in us\n");
	exit(0);
}

static int use_nanosleep;
static int priority;
static int num_threads = 1;
static int max_cycles;
static int clocksel = 2;
static int verbose;
static int interval = 1000;
static int distance = 500;

static int clocksources[] = {
	CLOCK_MONOTONIC,
	CLOCK_REALTIME,
	CLOCK_MONOTONIC_HR,
	CLOCK_REALTIME_HR
};

/* Process commandline options */
static void process_options (int argc, char *argv[])
{
	int error = 0;
	for (;;) {
		int option_index = 0;
		/** Options for getopt */
		static struct option long_options[] = {
			{"clock", required_argument, NULL, 'c'},
			{"distance", required_argument, NULL, 'd'},
			{"interval", required_argument, NULL, 'i'},
			{"loops", required_argument, NULL, 'l'},
			{"nanosleep", no_argument, NULL, 'n'},
			{"priority", required_argument, NULL, 'p'},
			{"threads", required_argument, NULL, 't'},
			{"verbose", no_argument, NULL, 'v'},
			{"help", no_argument, NULL, '?'},
			{NULL, 0, NULL, 0}
		};
		int c = getopt_long (argc, argv, "c:d:i:l:np:st:v",
			long_options, &option_index);
		if (c == -1)
			break;
		switch (c) {
		case 'c': clocksel = atoi(optarg); break;
		case 'd': distance = atoi(optarg); break;	
		case 'i': interval = atoi(optarg); break;	
		case 'l': max_cycles = atoi(optarg); break;
		case 'n': use_nanosleep = 1; break;
		case 'p': priority = atoi(optarg); break;
		case 't': num_threads = atoi(optarg); break;
		case 'v': verbose = 1; break;
		case '?': error = 1; break;
		}				
	}

	if (clocksel < 0 || clocksel > sizeof(clocksources))
		error = 1;

	if (priority < 0 || priority > 99)
		error = 1;

	if (num_threads < 1)
		error = 1;

	if (error)
		display_help ();
}

static void sighand(int sig)
{
	shutdown = 1;
}

static void print_stat(struct thread_param *par, int index, int verbose)
{
	struct thread_stat *stat = par->stats;
	
	if (!verbose) {
		printf("T:%2d (%5d) P:%2d I:%8ld C:%8lu Min:%8ld Act:%8ld Max:%8ld\n", 
		       index, stat->tid, par->prio, par->interval,
		       stat->cycles, stat->min, stat->act, stat->max);
	} else {
		while (stat->cycles != stat->cyclesread) {
			long diff = stat->values[stat->cyclesread & par->bufmsk];
			printf("%d:%lu:%ld\n", index, stat->cyclesread, diff);
			stat->cyclesread++;
		}
	}
}

int main(int argc, char **argv)
{
	sigset_t sigset;
	int signum = SIGUSR1;
	struct thread_param *par;
	struct thread_stat *stat;
	int i, ret = -1;

	if (geteuid()) {
		fprintf(stderr, "Need to run as root!\n");
		exit(-1);
	}

	process_options(argc, argv);

       	sigemptyset(&sigset);
       	sigaddset(&sigset, signum);
   	sigprocmask (SIG_BLOCK, &sigset, NULL);
	
	signal(SIGINT, sighand);
	signal(SIGTERM, sighand);
	
	par = calloc(num_threads, sizeof(struct thread_param));
	if (!par)
		goto out;
	stat = calloc(num_threads, sizeof(struct thread_stat));
	if (!stat)
		goto outpar;
	
	for (i = 0; i < num_threads; i++) {
		if (verbose) {
			stat[i].values = calloc(VALBUF_SIZE, sizeof(long));
			if (!stat[i].values)
				goto outall;
			par[i].bufmsk = VALBUF_SIZE - 1;
		}
		
		par[i].prio = priority;
		if (priority)
			priority--;
		par[i].clock = clocksources[clocksel];
		par[i].mode = use_nanosleep ? MODE_NANOSLEEP : MODE_CYCLIC;
		par[i].signal = signum;
		par[i].interval = interval;
		interval += distance;
		par[i].max_cycles = max_cycles;
		par[i].stats = &stat[i];
		stat[i].min = 1000000;
		stat[i].max = -1000000;
		pthread_create(&stat[i].thread, NULL, timerthread, &par[i]);
		stat[i].threadstarted = 1;
	}
	
	while (!shutdown) {
		char lavg[256];
		int fd, len, allstopped = 0;

		fd = open("/proc/loadavg", O_RDONLY, 0666);
		len = read(fd, &lavg, 255);
		close(fd);
		lavg[len-1] = 0x0;
		printf("%s          \n\n", lavg);

		for (i = 0; i < num_threads; i++) {
			
			print_stat(&par[i], i, verbose);
			
			if(stat[i].cycles < max_cycles)
				allstopped++;
		}
		usleep(10000);
		if (shutdown || allstopped == num_threads)
			break;
		if (!verbose)
			printf("\033[%dA", num_threads + 2);
	}
	
	ret = 0;
 outall:
	shutdown = 1;
	for (i = 0; i < num_threads; i++) {
		if (stat[i].threadstarted)
			pthread_join(stat[i].thread, NULL);
		if (stat[i].values)
			free(stat[i].values);
	}
	free(stat);
 outpar:
	free(par);
 out:
	exit(ret);
}
