/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Authors: Waiman Long <waiman.long@hp.com>
 *
 * This is the driver program for running the lock test.
 */
#define	_GNU_SOURCE
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <stdio.h>
#include <limits.h>
#include <locale.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <libgen.h>
#include <pthread.h>
#include <math.h>
#include <fcntl.h>
#include <sched.h>

/*
 * locktest sysfs files
 */
#define	SYSFS_ROOT	"/sys/kernel/locktest"
#define	SYSFS_CCOUNT	SYSFS_ROOT "/c_count"
#define	SYSFS_ICOUNT	SYSFS_ROOT "/i_count"
#define	SYSFS_LCOUNT	SYSFS_ROOT "/l_count"
#define	SYSFS_PCOUNT	SYSFS_ROOT "/p_count"
#define	SYSFS_RRATIO	SYSFS_ROOT "/rw_ratio"
#define	SYSFS_LOCKTYPE	SYSFS_ROOT "/locktype"
#define	SYSFS_LOADTYPE	SYSFS_ROOT "/loadtype"
#define	SYSFS_ETIME	SYSFS_ROOT "/etime"

/*
 * Lock types
 */
#define	LOCK_SPIN	0
#define	LOCK_RWLOCK	1
#define	LOCK_MUTEX	2
#define	LOCK_RWSEM	3
#define	LOCK_MAX	4
#define	LOCK_OTHER	LOCK_MAX

/*
 * Timing tests
 */
#define TIME_PAUSE	10
#define TIME_RMB	11
#define TIME_MB		12

/*
 * Load types
 */
#define	LOAD_STANDALONE	0
#define	LOAD_EMBEDDED	1

static char usage[] = "\
Usage: %s [-b|-B] [-h] [-v] [-c <load count>] [-i <iterations>]\n\
	[-l {spin|rw|mutex|rwsem} ] [-L {0|1}]\n\
	[-n <thread count>[-<max thread count>]]\n\
	[-p <pause count>] [-r <rw ratio>] [-t <time-test>]\n\
	[-s <cpu>] [-x <cpu-inc>]\n\
where -b - explicitly bind threads to different CPUs\n\
      -B - explicitly bind threads and show CPU numbers\n\
      -c - amount of load in critical section (default = 1)\n\
      -h - print this help message\n\
      -i - iteration count\n\
      -l - lock type (either spin or rw)\n\
      -L - load type (0: standalone, 1: embedded)\n\
      -n - thread count (a single value or a range)\n\
      -p - amount of pause between critical sections (default = 1)\n\
      -r - reader/writer ratio (for rwlock and rwsem)\n\
      -s - starting CPU number (default = 1)\n\
      -t - perform timing test (pause, mb, rmb)\n\
           other options ignored except -i\n\
      -v - verbose flag\n\
      -x - CPU number increment (default = 1)\n\
";

static int iterations = 5000000;	/* Iteration count	*/
static int rw_ratio   = 1;		/* Read:write ratio	*/
static int loadcnt    = 1;		/* Load count		*/
static int pausecnt   = 1;		/* Pause count		*/
static int threads    = 2;		/* Thread count		*/
static int maxthreads = 2;		/* Maximum thread count */
static int locktype   = LOCK_SPIN;	/* Lock type		*/
static int loadtype   = LOAD_STANDALONE;/* Load type		*/
static int startcpu   = 1;		/* Start CPU number	*/
static int cpuinc     = 1;		/* CPU number increment	*/
static int timetest;			/* Timing test type	*/
static int bindthread;			/* Bind thread to CPUs	*/
static int start;			/* 1 - start, -1 - exit */
static int verbose;

static char *lockname[LOCK_MAX] = {
	[LOCK_SPIN  ] = "spinlock",
	[LOCK_RWLOCK] = "rwlock",
	[LOCK_MUTEX ] = "mutex",
	[LOCK_RWSEM ] = "rwsem"
};

static struct tdata {
	pthread_t	pthread;	/* Pthread handle	 */
	int	 	done;		/* Done flag		 */
	unsigned long	etime;		/* Reported elapsed time */
} *tdata;

static void print_usage(char *cmd)
{
	fprintf(stderr, usage, cmd);
	exit(1);
}

static void write_sysfs(char *file, int value)
{
	char buf[80];
	int  len, fd;

	len = sprintf(buf, "%d", value);
	if ((fd = open(file, O_WRONLY)) < 0) {
		fprintf(stderr, "Error: Can't open %s!\n", file);
		exit(1);
	}
	if (write(fd, buf, len) != len) {
		fprintf(stderr, "Error: sysfs file %s write error!\n", file);
		exit(1);
	}
	close(fd);
}

static void *locktest_thread(void *dummy)
{
	int  tid = (long)dummy;
	char buf[80];
	int  retval = 0;
	int  fd;

	/*
	 * Wait until instructed to start
	 */
	while (start <= 0) {
		if (start < 0) {
			retval = -1;
			pthread_exit(&retval);
		}
		usleep(1);
	}
	if ((fd = open(SYSFS_ETIME, O_RDONLY)) < 0) {
		fprintf(stderr, "Thread %d error: Can't open " SYSFS_ETIME
			"!\n", tid);
		exit(1);
	}
	if (read(fd, buf, sizeof(buf)) <= 0) {
		fprintf(stderr, "Thread %d error: sysfs file " SYSFS_ETIME
			" read error!\n", tid);
		exit(1);
	}
	tdata[tid].etime = strtoul(buf, NULL, 10);
	tdata[tid].done  = 1;
	pthread_exit(&retval);
}

static void run_locktest(void)
{
	int i;
	unsigned long mean, min, max;
	double sd;	/* Standard deviation */
	double pcpurate;


	/*
	 * Set up the sysfs files
	 */
	write_sysfs(SYSFS_ICOUNT  , iterations);
	write_sysfs(SYSFS_CCOUNT  , threads);
	write_sysfs(SYSFS_RRATIO  , rw_ratio);
	write_sysfs(SYSFS_LOCKTYPE, locktype);
	write_sysfs(SYSFS_LOADTYPE, loadtype);
	write_sysfs(SYSFS_LCOUNT  , loadcnt);
	write_sysfs(SYSFS_PCOUNT  , pausecnt);

	/*
	 * Create the locktest threads
	 */
	for (i = 0 ; i < threads ; i++) {
		tdata[i].done  = 0;
		tdata[i].etime = 0;
		if (pthread_create(&tdata[i].pthread, NULL, locktest_thread,
				  (void *)(long)i)) {
			start = -1;
			perror("pthread_create");
			exit(1);
		}
	}
	if (bindthread) {
		cpu_set_t set;

		CPU_ZERO(&set);
		for (i = 0 ; i < threads ; i++) {
			int cpu = startcpu + cpuinc * i;

			CPU_SET(cpu, &set);
			if (pthread_setaffinity_np(tdata[i].pthread,
				sizeof(set), &set) < 0) {
				start = -1;
				perror("pthread_setaffinity_np");
				exit(1);
			}
			CPU_CLR(cpu, &set);
		}
		usleep(50);	/* Wait for the threads to be migrated */
	}
	start = 1;
	for (i = 0 ; i < threads ; i++)
		pthread_join(tdata[i].pthread, NULL);

	/*
	 * Compute the mean & standard deviation of the execution time
	 */
	min = INT_MAX;
	max = 0;
	for (i = 0, mean = 0 ; i < threads ; i++) {
		mean += tdata[i].etime;
		if (tdata[i].etime > max)
			max = tdata[i].etime;
		if (tdata[i].etime < min)
			min = tdata[i].etime;
	}
	mean = (mean + threads/2)/threads;

	for (i = 0, sd = 0.0 ; i < threads ; i++)
		sd += (tdata[i].etime - mean)*(tdata[i].etime - mean);
	sd = sqrt(sd/threads);

	if (bindthread > 1) {
		printf("CPUs = %d", startcpu);
		for (i = 1; i < threads; i++)
			printf("|%d", startcpu + i*cpuinc);
	} else {
		printf("Threads = %d", threads);
	}
	printf(", Min/Mean/Max = %'.1f/%'.1f/%'.1f ms, SD = %.2f [%'d]\n",
	       min/1000.0, mean/1000.0, max/1000.0, sd/1000, iterations);
	/*
	 * Compute per-cpu locking rate in kop/s
	 */
	pcpurate = ((double)iterations)/mean*1000000;
	printf("Threads = %d, Total Rate = %'.0f kop/s; "
	       "Percpu Rate = %'.0f kop/s\n",
	      threads, pcpurate * threads, pcpurate);
	if (verbose) {
		/*
		 * List the individual thread execution times
		 */
		for (i = 0 ; i < threads ; i++)
			printf("Thread %d: execution time = %'.1f ms\n",
				i, tdata[i].etime/1000.0);
	}

}

static void run_timetest(void)
{
	char buf[80];
	int  fd;
	unsigned long etime;
	double ns;

	/*
	 * Set up the sysfs files
	 */
	write_sysfs(SYSFS_ICOUNT  , iterations);
	write_sysfs(SYSFS_LOCKTYPE, timetest);

	if ((fd = open(SYSFS_ETIME, O_RDONLY)) < 0) {
		fprintf(stderr, "Error: Can't open " SYSFS_ETIME "!\n");
		exit(1);
	}
	if (read(fd, buf, sizeof(buf)) <= 0) {
		fprintf(stderr, "Error: sysfs file " SYSFS_ETIME
			" read error!\n");
		exit(1);
	}
	etime = strtoul(buf, NULL, 10);
	ns = ((double)etime) * 1000 / iterations;
	printf("%s timing test = %.2f ns\n",
	      (timetest == TIME_PAUSE) ? "pause" :
	      (timetest == TIME_RMB  ) ? "rmb"   : "mb", ns);
}

int main(int argc, char *argv[])
{
	int ret, c;
	char *cmd = argv[0];
	char *end;
	struct stat statbuf;

	setlocale(LC_NUMERIC, "en_US");
	while ((c = getopt(argc, argv, "bBc:hi:l:L:n:p:r:s:t:vx:")) != -1) {
		switch (c) {
		case 'b':
			bindthread = 1;
			break;
		case 'B':
			bindthread = 2;
			break;
		case 'c':
			loadcnt = atoi(optarg);
			break;
		case 'h':
			print_usage(cmd);
			break;
		case 'i':
			iterations = strtoul(optarg, &end, 0);
			if ((*end == 'M') || (*end == 'm'))
				iterations *= 1000000;
			else if ((*end == 'K') || (*end == 'k'))
				iterations *= 1000;
			break;
		case 'l':
			if (strcmp(optarg, "spin") == 0)
				locktype = LOCK_SPIN;
			else if (strcmp(optarg, "rw") == 0)
				locktype = LOCK_RWLOCK;
			else if (strcmp(optarg, "mutex") == 0)
				locktype = LOCK_MUTEX;
			else if (strcmp(optarg, "rwsem") == 0)
				locktype = LOCK_RWSEM;
			else if (isdigit(*optarg))
				locktype = atoi(optarg);
			else
				print_usage(cmd);
			break;
		case 'L':
			loadtype = atoi(optarg);
			break;
		case 'n':
			threads = strtoul(optarg, &end, 0);
			if (*end == '-') {
				maxthreads = strtoul(++end, NULL, 0);
				if (maxthreads < threads)
					print_usage(cmd);
			} else {
				maxthreads = threads;
			}
			break;
		case 'p':
			pausecnt = atoi(optarg);
			break;
		case 'r':
			rw_ratio = atoi(optarg);
			break;
		case 's':
			startcpu = atoi(optarg);
			break;
		case 't':
			if (strcmp(optarg, "pause") == 0)
				timetest = TIME_PAUSE;
			else if (strcmp(optarg, "rmb") == 0)
				timetest = TIME_RMB;
			else if (strcmp(optarg, "mb") == 0)
				timetest = TIME_MB;
			else
				print_usage(cmd);
			break;
		case 'v':
			verbose = 1;
			break;
		case 'x':
			cpuinc = atoi(optarg);
			break;
		default:
			print_usage(cmd);
			break;
		}
	}

	/*
	 * Check if the sysfs locktest directory is present
	 */
	if ((stat(SYSFS_ROOT, &statbuf) < 0) || !S_ISDIR(statbuf.st_mode)) {
		fprintf(stderr, "Error: " SYSFS_ROOT
			" directory doesn't exist!\n"
			"Please load the locktest kernel module.\n");
		exit(1);
	}

	/*
	 * Check if root is running it
	 */
	if (geteuid() != 0) {
		fprintf(stderr,
			"Error: this program needs to be run by root!\n");
		exit(1);
	}

	/*
	 * Print test parameters
	 */
	printf("\nRunning locktest with %s [iterations = %d",
	       (locktype < LOCK_MAX) ? lockname[locktype] : "other",
	       iterations);
	if ((locktype == LOCK_RWLOCK) || (locktype == LOCK_RWSEM))
		printf(", rw_ratio = %d:1", rw_ratio);
	printf("]\n");

	/*
	 * Increase scheduling priority to reduce run-to-run variation
	 */
	if (nice(-20) == -1) {
		perror("nice");
		exit(1);
	}

	/*
	 * Allocate thread data
	 */
	if ((tdata = (struct tdata *)calloc(sizeof(*tdata), maxthreads))
		   == NULL) {
		perror("calloc");
		exit(1);
	}

	if (timetest) {
		run_timetest();
		exit(0);
	}

	run_locktest();
	while (++threads <= maxthreads) {
		usleep(10);
		run_locktest();
	}
}
