#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/resource.h>
#include <stdlib.h>

static unsigned long page_size;

static void get_page_size(void)
{
	page_size = getpagesize();

	printf("Page size %lu\n", page_size);
}

static unsigned long count_mapped_pages(void)
{
	FILE *f = fopen("/proc/self/maps", "r");

	unsigned long start, end, ret = 0;

	while (fscanf(f, "%lx-%lx%*[^\n]", &start, &end) == 2) {
//		printf("%lx-%lx\n", start, end);

		ret += (end - start) / page_size;
	}

	printf("%lu pages mapped\n", ret);

	fclose(f);

	return ret;
}

static unsigned long get_vmem_rlimit(void)
{
	int ret;
	struct rlimit rlim;

	ret = getrlimit(RLIMIT_AS, &rlim);

	if (ret == -1) {
		perror("getrlimit failed");
		return 0;
	}

	printf("Soft limit %li, Hard limit %li\n",
	       rlim.rlim_cur, rlim.rlim_max);

	return rlim.rlim_cur;
}

static int set_vmem_rlimit(unsigned long size)
{
	int ret;
	struct rlimit rlim;
	
	ret = getrlimit(RLIMIT_AS, &rlim);

	if (ret == -1) {
		perror("getrlimit failed");
		return 1;
	}

	printf("Soft limit %li, Hard limit %li\n",
	       rlim.rlim_cur, rlim.rlim_max);

	rlim.rlim_cur = size;

	ret = setrlimit(RLIMIT_AS, &rlim);
	
	if (ret == -1) {
		perror("getrlimit failed");
		return 1;
	}

	get_vmem_rlimit();
	
	return 0;
}

static void *map_region(void *addr, unsigned long size)
{
	static int fd = -1;
	int flags = MAP_PRIVATE;
	void *res;

	if (fd == -1) {
		fd = open("/dev/zero", O_RDWR);

		if (fd == -1) {
			perror("open(/dev/zero)");
			return NULL;
		}
	}

	if (addr != NULL)
		flags |= MAP_FIXED;

	printf("Mapping region addr %p size %lu\n", addr, size);

	res = mmap(addr, size, PROT_READ | PROT_WRITE, flags, fd, 0);

	if (res == MAP_FAILED) {
		perror("mmap failed");
		return NULL;
	}

	printf("Mapped at addr %p\n", res);

	return res;
}

static void unmap_region(void *addr, unsigned long len)
{
	int ret;

	printf("Unmapping %p\n", addr);

	ret = munmap(addr, len);

	if (ret < 0)
		perror("munmap failed");
}

/*
 * Test 1 (Purely overlapping mappings when ulimit is reached works)
 *
 * Do a mapping.
 *
 * Set rlimt to current mapped size.
 *
 * Do overlapping mapping -> should work.
 *
 * Do no-overlaping mapping -> shouldn't work.
 */
static int test1(void)
{
	void *addr;

	count_mapped_pages();
	
	addr = map_region(NULL, page_size + 10);

	set_vmem_rlimit(count_mapped_pages() * page_size);

	/* should work */
	addr = map_region(addr, page_size + 10);

	if (addr == NULL) {
		printf("Overlapping mmap failed while it shouldn't\n");
		return 1;
	}

	/* shouldn't work */
	addr = map_region(NULL, page_size + 10);

	if (addr != NULL) {
		printf("Non-overlapping mapping succeeded while it shouldn't\n");
		return 1;
	}

	count_mapped_pages();
	
	printf("Test PASSED\n");
	
	return 0;
}

/*
 * Test 2 (Partially overlapping mapping over ulimit fails correctly)
 *
 * Create mapping
 *
 * Set rlimit
 *
 * Do overlapping mapping over the limit
 *
 * Check that original mapping is preserved
 */
static int test2(void)
{
	void *addr;
	unsigned long mapped_pages;

	count_mapped_pages();

	addr = map_region(NULL, 10 * page_size + 43);

	/* Set rlimit less than currently mapped pages */
	set_vmem_rlimit(count_mapped_pages() * page_size - 4 * page_size);

	/*
	 * Punch a hole into the mapping, rlimit should be equal to
	 * mapped memory now
	 */
	unmap_region((char *)addr + page_size, 4 * page_size);
	
	mapped_pages = count_mapped_pages();

	/* should return enomem */
	addr = map_region(addr, 10 * page_size + 1);

	if (mapped_pages != count_mapped_pages()) {
		printf("Failed mapping unmapped the pages\n");
		return 1;
	}

	printf("Test PASSED\n");

	return 0;
}

/* 
 * Prepares mapping with some holes
 *
 * The map should now look like:
 *  0                   21
 *  X    X  XXXX       XX
 *
 * And we should have two pages left to the limit.
 */
static void *prepare_memmap(void)
{
	void *addr;

	count_mapped_pages();

	/* Map 21 pages */
	addr = map_region(NULL, 20 * page_size+1);
	
	/* Set rlimit less than currently mapped pages */
	set_vmem_rlimit((count_mapped_pages() - 6) * (page_size - 1));
	
	unmap_region(addr, page_size - 1);
	unmap_region((char*)addr + 5 * page_size, page_size - 1);
	unmap_region((char*)addr + 8 * page_size, 5 * (page_size - 1));
	unmap_region((char*)addr + 19 * page_size, 2 * (page_size - 1));

	count_mapped_pages();

	return addr;
}

/*
 * Test 3 (Check that accouting is done right)
 *
 * Try to map over the 21 pages, we will hit all
 * the holes => we are over limit.
 */
static int test3(void)
{
	void *addr;

	addr = prepare_memmap();

	if (addr == NULL)
		return 1;

	/* this should fail now */
	addr = map_region(addr, 20 * page_size + 3);

	if (addr != NULL) {
		printf("Mapping over the limit succeded\n");
		return 1;
	}

	printf("Test PASSED\n");
	return 0;
}

/*
 * Test 4 (Check that accouting is done right)
 *
 * Try to map start of the mapping, we should
 * be fine after overlapps are unmapped.
 */
static int test4(void)
{
	void *addr;

	addr = prepare_memmap();
	
	/* this should work */
	addr = map_region(addr, 8 * page_size);

	if (addr == NULL) {
		printf("Mapping failed unexpectedly\n");
		return 1;
	}

	printf("Test PASSED\n");
	return 0;
}
	
/*
 * Test 5 (Check that accouting is done right)
 *
 * Try to map start of the mapping we are one page
 * over the limit after unmapping the overlaps.
 */
static int test5(void)
{
	void *addr;

	addr = prepare_memmap();
	
	/* this should fail */
	addr = map_region(addr, 10 * (page_size - 1));
	
	if (addr != NULL) {
		printf("Mapping over the limit succeded\n");
		return 1;
	}
	
	printf("Test PASSED\n");
	return 0;
}

/*
 * Test 6 (Check that accouting is done right)
 *
 * Map pages in the middle of the mapping, we
 * should be fine after overlaps are unmapped.
 */
static int test6(void)
{
	void *addr;

	addr = prepare_memmap();
	
	/* this should fail */
	addr = map_region((char*)addr + 9 * page_size, 11 * (page_size - 1));
	
	if (addr != NULL) {
		printf("Mapping over the limit succeded\n");
		return 1;
	}
	
	printf("Test PASSED\n");
	return 0;
}

/*
 * Test 7 (Check that accouting is done right)
 *
 * Map pages at the end of the mapping, we
 * should be fine after overlaps are unmapped.
 */
static int test7(void)
{
	void *addr;

	addr = prepare_memmap();
	
	/* this should work */
	addr = map_region((char*)addr + 9 * page_size, 10 * (page_size - 1));
	
	if (addr != NULL) {
		printf("Mapping over the limit succeded\n");
		return 1;
	}
	
	printf("Test PASSED\n");
	return 0;
}

int main(int argc, char *argv[])
{
	if (argc != 2) {
		printf("usage: %s 1-4\n", argv[0]);
		return 1;
	}
	
	get_page_size();

	switch (atoi(argv[1])) {
	case 1:
		return test1();
	case 2:
		return test2();
	case 3:
		return test3();
	case 4:
		return test4();
	case 5:
		return test5();
	case 6:
		return test6();
	case 7:
		return test7();
	}

	return 0;
}
