#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <string.h>
#include <elf.h>
#include <getopt.h>
#include <unistd.h>

enum {
	FALSE = 0,
	TRUE = 1,
};

enum {
	MAX_DEBUG_LEVEL = 2,
};

enum {
	PAGE_SIZE = (4 << 10),
	DEFAULT_BLOCK_SIZE = PAGE_SIZE,
};

#define DEFAULT_MEMORY_DEVICE "/dev/oldmem"

static void usage(void);
static int check_elf_format(FILE *dumpfp);
static int read_elf_headers(FILE *dumpfp, Elf64_Ehdr **ehdrpp,
			    Elf64_Phdr **phdrpp, char **notep);
static int write_elf_headers(FILE *outfp, Elf64_Ehdr *ehdrp,
			     Elf64_Phdr *phdrp);
static int write_elf_segments(FILE *outfp, FILE *memdevfp, Elf64_Ehdr *ehdrp,
			      Elf64_Phdr *phdrp, const char *note);

static struct option longopts[] = {
        {"blocksize", required_argument, NULL, 'b'},
	{"memorydevice", required_argument, NULL, 'm'},
	{"debug", required_argument, NULL, 'd'},
        {"help", no_argument, NULL, 'h'},
        {0, 0, 0, 0}
};

static int debug;
static size_t blocksize = DEFAULT_BLOCK_SIZE;
static char *memory_device = DEFAULT_MEMORY_DEVICE;

int main(int argc, char **argv)
{
	char *dumpfile = NULL, *outfile = NULL;
	char *notep = NULL;
	FILE *dumpfp = NULL, *outfp = NULL, *memdevfp = NULL;
	Elf64_Ehdr *ehdrp = NULL;
	Elf64_Phdr *phdrp = NULL;
	int opt, iself, retval = EXIT_FAILURE;

        while ((opt = getopt_long(argc, argv, "b:d:hm:", longopts,
				  NULL)) != -1) {
                switch (opt) {
                case 'b':
			blocksize = strtoul(optarg, NULL, 10);
                        break;
		case 'd':
			debug = strtol(optarg, NULL, 10);
			if (debug > MAX_DEBUG_LEVEL)
				debug = MAX_DEBUG_LEVEL;
			break;
		case 'h':
			usage();
			exit(0);
		case 'm':
			memory_device = optarg;
			break;
		default:
			usage();
			exit(1);
			break;
		}
	}

	if (!argv[optind] || !argv[optind+1]) {
		usage();
		exit(1);
	}

	dumpfile = argv[optind];
	outfile = argv[optind+1];

	if (debug) {
		printf("block size: %lu\n", blocksize);
		printf("memory device: %s\n", memory_device);
	}

	dumpfp = fopen(dumpfile, "r");
	if (!dumpfp) {
		perror("fopen");
		goto error;
	}

	memdevfp = fopen(memory_device, "r");
	if (!memdevfp) {
		perror("fopen");
		goto error;
	}

	outfp = fopen(outfile, "w");
	if (!outfp) {
		perror("fopen");
		goto error;
	}

	iself = check_elf_format(dumpfp);
	if (iself < 0)
		goto error;
	if (!iself) {
		fprintf(stderr, "%s is not ELF file\n", dumpfile);
		goto error;
	}

	if (read_elf_headers(dumpfp, &ehdrp, &phdrp, &notep))
		goto error;

	if (write_elf_headers(outfp, ehdrp, phdrp))
		goto error;

	if (write_elf_segments(outfp, memdevfp, ehdrp, phdrp, notep))
		goto error;

	retval = EXIT_SUCCESS;
error:
	if (dumpfp)
		fclose(dumpfp);
	if (outfp)
		fclose(outfp);
	if (memdevfp)
		fclose(memdevfp);
	if (ehdrp)
		free(ehdrp);
	if (phdrp)
		free(phdrp);
	return retval;
}

static void usage(void)
{
	printf("usage: copyvmcore [--help] [--debug=<debug level>] [--blocksize=<block size>] <vmcore> <outfile>\n");
}

static int check_elf_format(FILE *dumpfp)
{
	char magic[SELFMAG];

	if (fread(magic, sizeof(char), SELFMAG, dumpfp) != SELFMAG) {
		perror("fread");
		return -1;
	}

	return strncmp((const char *)magic, ELFMAG, SELFMAG) == 0;
}

static int read_elf_headers(FILE *dumpfp, Elf64_Ehdr **ehdrpp, Elf64_Phdr **phdrpp, char **notepp)
{
	Elf64_Ehdr *ehdrp = NULL;
	Elf64_Phdr *phdrp = NULL;
	char *notep = NULL;
	int i, retval = -1;

	ehdrp = malloc(sizeof(*ehdrp));
	if (!ehdrp) {
		perror("malloc");
		goto error;
	}

	if (fseek(dumpfp, 0, SEEK_SET) < 0) {
		perror("fseek");
		goto error;
	}

	if (fread(ehdrp, sizeof(*ehdrp), 1, dumpfp) != 1) {
		perror("fread");
		goto error;
	}

	phdrp = calloc(ehdrp->e_phnum, ehdrp->e_phentsize);
	if (!phdrp) {
		perror("calloc");
		goto error;
	}

	if (fseek(dumpfp, ehdrp->e_phoff, SEEK_SET) < 0) {
		perror("fseek");
		goto error;
	}

	if (fread(phdrp, ehdrp->e_phentsize * ehdrp->e_phnum, 1, dumpfp) != 1) {
		perror("fread");
		goto error;
	}

	for (i = 0; i < (long)ehdrp->e_phnum; ++i) {
		unsigned long start, end, size, offset;

		if (phdrp[i].p_type != PT_NOTE)
			continue;

		start = phdrp[i].p_paddr;
		size = phdrp[i].p_filesz;
		offset = phdrp[i].p_offset;
		end = start + size;

		if (fseek(dumpfp, offset, SEEK_SET) < 0) {
			perror("fseek");
			goto error;
		}

		notep = malloc(size);
		if (!notep) {
			perror("malloc");
			goto error;
		}

		if (fread(notep, size, 1, dumpfp) != 1) {
			perror("fread");
			goto error;
		}

		/* assum only one PT_NOTE entry. */
		break;
	}

	*ehdrpp = ehdrp;
	*phdrpp = phdrp;
	*notepp = notep;

	retval = 0;
error:
	if (retval) {
		if (phdrp)
			free(phdrp);
		if (ehdrp)
			free(ehdrp);
		if (notep)
			free(notep);
	}
	return retval;
}

static int
write_elf_headers(FILE *outfp, Elf64_Ehdr *ehdrp, Elf64_Phdr *phdrp)
{
	int i;

	if (fwrite(ehdrp, sizeof(*ehdrp), 1, outfp) != 1) {
		perror("fwrite");
		return -1;
	}
	for (i = 0; i < ehdrp->e_phnum; ++i) {
		if (fwrite(&phdrp[i], sizeof(phdrp[i]), 1, outfp) != 1) {
			perror("fwrite");
			return -1;
		}
	}
	return 0;
}

static int
write_elf_segments(FILE *outfp, FILE *memdevfp, Elf64_Ehdr *ehdrp,
		   Elf64_Phdr *phdrp, const char *note)
{
	int i, retval = -1;

	for (i = 0; i < (long)ehdrp->e_phnum; ++i) {
		unsigned long type, start, end, size, offset;
		char *buf = MAP_FAILED;

		type = phdrp[i].p_type;

		if (type != PT_LOAD && type != PT_NOTE)
			continue;

		start = phdrp[i].p_paddr;
		size = phdrp[i].p_filesz;
		offset = phdrp[i].p_offset;
		end = start + size;

		if (debug)
			printf("%s: %016lx-%016lx (%lu MB)\n",
			       type == PT_LOAD ? "LOAD" : "NOTE", start, end,
			       size >> 20);

		if (fseek(outfp, offset, SEEK_SET) < 0) {
			perror("fseek");
			goto error;
		}

		if (type == PT_NOTE) {
			if (fwrite(note, size, 1, outfp) != 1) {
				perror("fwrite");
				goto error;
			}
		} else if (type == PT_LOAD) {
			unsigned long p;

			for (p = start; p < end; p += blocksize) {
				size_t mapped_size = size >= blocksize ? blocksize : size;

				if (debug)
					printf("MAP: %016lx-%016lx (%lu)\n", p, p + mapped_size,
					       mapped_size);

				buf = mmap(NULL, mapped_size, PROT_READ, MAP_PRIVATE,
					   fileno(memdevfp), p);
				if (buf == MAP_FAILED) {
					perror("mmap");
					goto error;
				}

				if (fwrite(buf, mapped_size, 1, outfp) != 1) {
					perror("fwrite");
					munmap(buf, mapped_size);
					goto error;
				}

				munmap(buf, mapped_size);

				size -= mapped_size;
			}
		}
	}

	retval = 0;
error:
	return retval;
}
