#include <fcntl.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include "kvm.h"

struct vm *create_vm(void)
{
	struct vm *vm = wmalloc(sizeof(*vm));

	vm->sys_fd = open("/dev/kvm", O_RDWR);
	if (vm->sys_fd < 0)
		die("open /dev/kvm fail.\n");

	vm->fd = ioctl(vm->sys_fd, KVM_CREATE_VM, 0);
	if (vm->fd < 0)
		die("KVM_CREATE_VM fail.\n");
	
//	if (ioctl(vm->fd, KVM_SET_TSS_ADDR, 0xfffbd000) < 0)
//		die("KVM_SET_TSS_ADDR ioctl");

	return vm;
}

#define KVM_MEM_MAX_NR_PAGES ((1UL << 31) - 1)
#define KVM_MEM_MAX_SIZE (pfn2pa(KVM_MEM_MAX_NR_PAGES))

void  kvm_register_mem_slot(struct vm *vm, u64 gpa, void *host, u64 size, int dirty_log, int slot)
{
	struct kvm_userspace_memory_region mem;
	int ret;

	mem = (struct kvm_userspace_memory_region) {
		.slot			= slot,
		.guest_phys_addr	= gpa,
		.memory_size		= size,
		.userspace_addr 	= (u64)(unsigned long)host,
	};

	if (dirty_log)
		mem.flags = KVM_MEM_LOG_DIRTY_PAGES;

	dprintf("slot:%d, gpa:%llx, host:%llx, size:%llx flags:%x.\n", mem.slot, mem.guest_phys_addr,
		 mem.userspace_addr, mem.memory_size, mem.flags);

	ret = ioctl(vm->fd, KVM_SET_USER_MEMORY_REGION, &mem);
	if (ret < 0) {
		perror("KVM_SET_USER_MEMORY_REGION");
		die("KVM_SET_USER_MEMORY_REGION ioctl\n");
	}
}

void kvm_register_mem(struct vm *vm, u64 gpa, void *host, u64 size)
{
	kvm_register_mem_slot(vm, gpa, host, size, 0, vm->slot++);
}

void kvm_register_mem_dirty(struct vm *vm, u64 gpa, void *host, u64 size)
{
	kvm_register_mem_slot(vm, gpa, host, size, 1, vm->slot++);
}

static void setup_sregs(struct vcpu *vcpu)
{
//	unsigned long gsbase;
	struct kvm_sregs sregs;
	struct kvm_segment dseg;
	struct kvm_segment cseg;

	memset(&sregs, 0, sizeof(sregs));
	memset(&dseg, 0, sizeof(dseg));
	memset(&cseg, 0, sizeof(cseg));

    dseg.base = 0; dseg.limit = -1U; dseg.type = 3; dseg.present = 1;
    dseg.dpl = 3; dseg.db = 1; dseg.s = 1; dseg.l = 0; dseg.g = 1;

	cseg = dseg;
	cseg.type = 11;

    sregs.cs = cseg; asm ("mov %%cs, %0" : "=rm"(sregs.cs.selector));
    sregs.ds = dseg; asm ("mov %%ds, %0" : "=rm"(sregs.ds.selector));
    sregs.es = dseg; asm ("mov %%es, %0" : "=rm"(sregs.es.selector));
    sregs.fs = dseg; asm ("mov %%fs, %0" : "=rm"(sregs.fs.selector));
    sregs.gs = dseg; asm ("mov %%gs, %0" : "=rm"(sregs.gs.selector));
    sregs.ss = dseg; asm ("mov %%ss, %0" : "=rm"(sregs.ss.selector));

//    asm ("mov %%gs:0, %0" : "=r"(gsbase));
 //   sregs.gs.base = gsbase;

    sregs.tr.base = (long)vcpu->stack + 5 * PAGE_SIZE;
    sregs.tr.type = 11;
    sregs.tr.s = 0;
    sregs.tr.present = 1;

    sregs.cr0 = 0x11; /* PE, ET, !PG */
    sregs.cr4 = 0;
    sregs.efer = 0;
 //	sregs.efer = (1 << 7) | (1 << 10);
    sregs.apic_base = 0xfee00000;

   if (ioctl(vcpu->fd, KVM_SET_SREGS, &sregs) < 0)
		die("KVM_SET_SREGS failed.\n");
}

static void setup_regs(struct vcpu *vcpu, void(*fun)(void))
{
	struct kvm_regs regs;
//	unsigned long * sp;

	memset(&regs, 0, sizeof(regs));

	regs.rflags = 0x3202;

	regs.rsp = (long)vcpu->stack + 2 * PAGE_SIZE;
	regs.rbp = (long)vcpu->stack + 3 * PAGE_SIZE;

	regs.rsi = (u64)vcpu->id;
	regs.rip = (long)fun;

	dprintf("rip %llx\n", regs.rip);

	if (ioctl(vcpu->fd, KVM_SET_REGS, &regs) < 0) {
		perror("KVM_SET_REGS");
		die("KVM_SET_REGS failed.\n");
	}
}

void vcpu_register_fun(struct vcpu *vcpu, void(*fun)(void))
{
	setup_sregs(vcpu);
	setup_regs(vcpu, fun);
}

struct vcpu *create_vcpu(struct vm *vm, int vcpu_id)
{
	int mmap_size;
	struct vcpu *vcpu = wmalloc(sizeof(*vcpu));

	vcpu->vm = vm;
	vcpu->id = vcpu_id;
	vcpu->fd = ioctl(vm->fd, KVM_CREATE_VCPU, vcpu_id);
	if (vcpu->fd < 0)
		die("KVM_CREATE_VCPU ioctl.\n");

	mmap_size = ioctl(vm->sys_fd, KVM_GET_VCPU_MMAP_SIZE, 0);
	if (mmap_size < 0)
		die("KVM_GET_VCPU_MMAP_SIZE ioctl.\n");

	vcpu->kvm_run = mmap(NULL, mmap_size, PROT_READ | PROT_WRITE, MAP_SHARED, vcpu->fd, 0);
	if (vcpu->kvm_run == MAP_FAILED)
		die("mmap vcpu fail.\n");

	return vcpu;
}

void vcpu_run(struct vcpu *vcpu)
{
	int err;

	err = ioctl(vcpu->fd, KVM_RUN, 0);
	if (err && (errno != EINTR && errno != EAGAIN)) {
		perror("RUN");
		fmt_die("VCPU %d run failed.\n", vcpu->id);
	}	
}


