/*
 * $Id: amd7409.c,v 1.5 2001/01/18 21:56:60 vojtech Exp $
 *
 *  Copyright (c) 2000-2001 Vojtech Pavlik
 *
 *  Based on the work of:
 *	Andre Hedrick
 *
 *  Sponsored by SuSE
 */

/*
 * AMD 755/756/766 IDE driver for Linux.
 *
 * UDMA66 and higher modes are autodetected only in case the BIOS has detected
 * a 80 wire cable. To ignore the BIOS data, use 'ide0=ata66' or 'ide1=ata66'
 * on the kernel command line.
 *
 * For correct operation it's needed to tell the driver the speed of the PCI
 * bus. The default, and most common value is 33 MHz. Unless you're
 * overclocking, your system is using this value and you don't need to use any
 * command line options. If you run your PCI bus speed at 25, 30, 37, 40 or 42
 * MHz, use the 'idebus=xx' option of the IDE driver. Note that this has
 * nothing to do with UDMA66.
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Should you need to contact me, the author, you can do so either by
 * e-mail - mail your message to <vojtech@suse.cz>, or by paper mail:
 * Vojtech Pavlik, Ucitelska 1576, Prague 8, 182 00 Czech Republic
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/blkdev.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/ide.h>
#include <asm/io.h>

#include "ide-timing.h"

#define AMD_IDE_ENABLE		0x40
#define AMD_IDE_CONFIG		0x41
#define AMD_CABLE_DETECT	0x42
#define AMD_DRIVE_TIMING	0x48
#define AMD_8BIT_TIMING		0x4e
#define AMD_ADDRESS_SETUP	0x4c
#define AMD_UDMA_TIMING		0x50

#define AMD_UDMA		0x07
#define AMD_UDMA_33		0x01
#define AMD_UDMA_66		0x02
#define AMD_UDMA_100		0x04
#define AMD_NO_SWDMA		0x08

#ifndef PCI_DEVICE_ID_AMD_VIPPL_7411
#define PCI_DEVICE_ID_AMD_VIPPL_7411	0x7411
#endif

/*
 * AMD SouthBridge chips.
 */

static struct amd_ide_chip {
	char *name;
	unsigned short id;
	unsigned char rev;
	unsigned char flags;
} amd_ide_chips[] = {
	{ "766 ViperPlus",	PCI_DEVICE_ID_AMD_VIPPL_7411, 0x00, AMD_UDMA_100 },
	{ "756/c4+ Viper",	PCI_DEVICE_ID_AMD_VIPER_7409, 0x07, AMD_UDMA_66 },
	{ "756 Viper",		PCI_DEVICE_ID_AMD_VIPER_7409, 0x00, AMD_UDMA_66 | AMD_NO_SWDMA },
	{ "755 Cobra",		PCI_DEVICE_ID_AMD_COBRA_7401, 0x00, AMD_UDMA_33 | AMD_NO_SWDMA },
	{ NULL }
};

static struct amd_ide_chip *amd_config;
static unsigned char amd_enabled;
static unsigned int amd_80w;
static unsigned int amd_clock;

static unsigned char amd_cyc2udma[] = { 6, 6, 5, 4, 0, 1, 1, 2, 2, 3, 3 };
static unsigned char amd_udma2cyc[] = { 4, 6, 8, 10, 3, 2, 1, 1 };

/*
 * AMD /proc entry.
 */

#ifdef CONFIG_PROC_FS

#include <linux/stat.h>
#include <linux/proc_fs.h>

int amd7409_proc, amd_base;
static struct pci_dev *bmide_dev;
extern int (*amd7409_display_info)(char *, char **, off_t, int); /* ide-proc.c */

#define amd_print(format, arg...) p += sprintf(p, format "\n" , ## arg)
#define amd_print_drive(name, format, arg...)\
	p += sprintf(p, name); for (i = 0; i < 4; i++) p += sprintf(p, format, ## arg); p += sprintf(p, "\n");

static int amd_get_info(char *buffer, char **addr, off_t offset, int count)
{
	short clk100, speed[4], cycle[4], setup[4], active[4],
		recover[4], uen[4], udma[4], active8b[4], recover8b[4];
	struct pci_dev *dev = bmide_dev;
	unsigned int v, u, i;
	unsigned short c, w;
	unsigned char t;
	char *p = buffer;

	amd_print("----------AMD BusMastering IDE Configuration----------------");

	amd_print("Driver Version:                     1.5");
	amd_print("South Bridge:                       AMD-%s", amd_config->name);

	pci_read_config_byte(dev, PCI_REVISION_ID, &t);
	amd_print("Revision:                           IDE %#x", t);

	amd_print("BM-DMA base:                        %#x", amd_base);
	amd_print("PCI clock:                          %dMHz", amd_clock);
	
	pci_read_config_byte(dev, AMD_IDE_CONFIG, &t);
	clk100 = ((amd_config->flags & AMD_UDMA) == AMD_UDMA_100) && (t & 4);

	amd_print("UDMA100 clock source:                %s", 
		((amd_config->flags & AMD_UDMA) == AMD_UDMA_100) ?
			(clk100 ? "CLK100 Pin" : "PCI PhaseLock") : "N/A");

	amd_print("-----------------------Primary IDE-------Secondary IDE------");

	amd_print("Prefetch Buffer:       %10s%20s", (t & 0x80) ? "yes" : "no", (t & 0x20) ? "yes" : "no");
	amd_print("Post Write Buffer:     %10s%20s", (t & 0x40) ? "yes" : "no", (t & 0x10) ? "yes" : "no");

	pci_read_config_byte(dev, AMD_IDE_ENABLE, &t);
	amd_print("Enabled:               %10s%20s", (t & 0x02) ? "yes" : "no", (t & 0x01) ? "yes" : "no");

	c = inb(amd_base + 0x02) | (inb(amd_base + 0x0a) << 8);
	amd_print("Simplex only:          %10s%20s", (c & 0x80) ? "yes" : "no", (c & 0x8000) ? "yes" : "no");

	amd_print("Cable Type:            %10s%20s", (amd_80w & 1) ? "80w" : "40w", (amd_80w & 2) ? "80w" : "40w");

	amd_print("-------------------drive0----drive1----drive2----drive3-----");

	pci_read_config_byte(dev, AMD_ADDRESS_SETUP, &t);
	pci_read_config_dword(dev, AMD_DRIVE_TIMING, &v);
	pci_read_config_word(dev, AMD_8BIT_TIMING, &w);
	pci_read_config_dword(dev, AMD_UDMA_TIMING, &u);

	for (i = 0; i < 4; i++) {
		setup[i]     = ((t >> ((3 - i) << 1)) & 0x3) + 1;
		recover8b[i] = ((w >> ((1 - (i >> 1)) << 3)) & 0xf) + 1;
		active8b[i]  = ((w >> (((1 - (i >> 1)) << 3) + 4)) & 0xf) + 1;
		active[i]    = ((v >> (((3 - i) << 3) + 4)) & 0xf) + 1;
		recover[i]   = ((v >> ((3 - i) << 3)) & 0xf) + 1;

		udma[i] = amd_udma2cyc[((u >> ((3 - i) << 3)) & 0x7)];
		uen[i]  = ((u >> ((3 - i) << 3)) & 0x40) ? 1 : 0;

		if (udma[i] == 1) {
			speed[i] = clk100 ? 1000 : amd_clock * 30;
			cycle[i] = clk100 ? 20 : 666 / amd_clock;
			continue;
		}

		speed[i] = 40 * amd_clock / (uen[i] ? udma[i] : (active[i] + recover[i]) * 2);
		cycle[i] = 1000 / amd_clock * (uen[i] ? udma[i] : (active[i] + recover[i]) * 2) / 2;
	}

	amd_print_drive("Transfer Mode: ", "%10s",
			(c & ((i & 1) ? 0x40 : 0x20) << ((i & 2) << 2)) ? (uen[i] ? "UDMA" : "DMA") : "PIO");

	amd_print_drive("Address Setup: ", "%8dns", (1000 / amd_clock) * setup[i]);
	amd_print_drive("Cmd Active:    ", "%8dns", (1000 / amd_clock) * active8b[i]);
	amd_print_drive("Cmd Recovery:  ", "%8dns", (1000 / amd_clock) * recover8b[i]);
	amd_print_drive("Data Active:   ", "%8dns", (1000 / amd_clock) * active[i]);
	amd_print_drive("Data Recovery: ", "%8dns", (1000 / amd_clock) * recover[i]);
	amd_print_drive("Cycle Time:    ", "%8dns", cycle[i]);
	amd_print_drive("Transfer Rate: ", "%4d.%dMB/s", speed[i] / 10, speed[i] % 10);

	return p - buffer;	/* hoping it is less than 4K... */
}

#endif

/*
 * amd_set_speed() writes timing values to the chipset registers
 */

static void amd_set_speed(struct pci_dev *dev, unsigned char dn, struct ide_timing *timing)
{
	unsigned char t;

	pci_read_config_byte(dev, AMD_ADDRESS_SETUP, &t);
	t = (t & ~(3 << ((3 - dn) << 1))) | ((FIT(timing->setup, 1, 4) - 1) << ((3 - dn) << 1));
	pci_write_config_byte(dev, AMD_ADDRESS_SETUP, t);

	pci_write_config_byte(dev, AMD_8BIT_TIMING + (1 - (dn >> 1)),
		((FIT(timing->act8b, 1, 16) - 1) << 4) | (FIT(timing->rec8b, 1, 16) - 1));

	pci_write_config_byte(dev, AMD_DRIVE_TIMING + (3 - dn),
		((FIT(timing->active, 1, 16) - 1) << 4) | (FIT(timing->recover, 1, 16) - 1));

	switch (amd_config->flags & AMD_UDMA) {
		case AMD_UDMA_33:  t = timing->udma ? (0xc0 | (FIT(timing->udma, 2, 5) - 2)) : 0x03; break;
		case AMD_UDMA_66:  t = timing->udma ? (0xc0 | amd_cyc2udma[FIT(timing->udma, 2, 10)]) : 0x03; break;
		case AMD_UDMA_100: t = timing->udma ? (0xc0 | amd_cyc2udma[FIT(timing->udma, 1, 10)]) : 0x03; break;
	}

	pci_write_config_byte(dev, AMD_UDMA_TIMING + (3 - dn), t);
}

/*
 * amd_set_drive() computes timing values configures the drive and
 * the chipset to a desired transfer mode. It also can be called
 * by upper layers.
 */

static int amd_set_drive(ide_drive_t *drive, unsigned char speed)
{
	ide_drive_t *peer = HWIF(drive)->drives + (~drive->dn & 1);
	struct ide_timing t, p;
	unsigned char c;
	int T, UT;

	if (speed != XFER_PIO_SLOW && speed != drive->current_speed)
		if (ide_config_drive_speed(drive, speed))
			printk(KERN_WARNING "ide%d: Drive %d didn't accept speed setting. Oh, well.\n",
				drive->dn >> 1, drive->dn & 1);

	T = 1000 / amd_clock;
	UT = T / MIN(MAX(amd_config->flags & AMD_UDMA, 1), 2);

	ide_timing_compute(drive, speed, &t, T, UT);

	if (peer->present) {
		ide_timing_compute(peer, peer->current_speed, &p, T, UT);
		ide_timing_merge(&p, &t, &t, IDE_TIMING_8BIT);
	}

	if (speed == XFER_UDMA_5) {
		pci_read_config_byte(HWIF(drive)->pci_dev, AMD_IDE_CONFIG, &c);
		if ((~c & 4) && (amd_clock <= 33)) t.udma = 1;	/* UDMA100 clock OK */
	}

	amd_set_speed(HWIF(drive)->pci_dev, drive->dn, &t);

	if (!drive->init_speed)	
		drive->init_speed = speed;
	drive->current_speed = speed;

	return 0;
}

/*
 * amd7409_tune_drive() is a callback from upper layers for
 * PIO-only tuning.
 */

static void amd7409_tune_drive(ide_drive_t *drive, unsigned char pio)
{
	if (!((amd_enabled >> HWIF(drive)->channel) & 1))
		return;

	if (pio == 255) {
		amd_set_drive(drive, ide_find_best_mode(drive, XFER_PIO | XFER_EPIO));
		return;
	}

	amd_set_drive(drive, XFER_PIO_0 + MIN(pio, 5));
}

#ifdef CONFIG_BLK_DEV_IDEDMA

/*
 * amd7409_dmaproc() is a callback from upper layers that can do
 * a lot, but we use it for DMA/PIO tuning only, delegating everything
 * else to the default ide_dmaproc().
 */

int amd7409_dmaproc(ide_dma_action_t func, ide_drive_t *drive)
{

	if (func == ide_dma_check) {

		short w80 = HWIF(drive)->udma_four;

		short speed = ide_find_best_mode(drive,
			XFER_PIO | XFER_EPIO | XFER_MWDMA | XFER_UDMA |
			((amd_config->flags & AMD_NO_SWDMA) ? 0 : XFER_SWDMA) |
			(w80 && (amd_config->flags & AMD_UDMA) == AMD_UDMA_66 ? XFER_UDMA_66 : 0) |
			(w80 && (amd_config->flags & AMD_UDMA) == AMD_UDMA_100 ? XFER_UDMA_100 : 0));

		func = ((speed & XFER_MODE) != XFER_PIO) ? ide_dma_on : ide_dma_off_quietly;

		amd_set_drive(drive, speed);
	}

	return ide_dmaproc(func, drive);
}

#endif /* CONFIG_BLK_DEV_IDEDMA */

/*
 * The initialization callback. Here we determine the IDE chip type
 * and initialize its drive independent registers.
 */

unsigned int __init pci_init_amd7409(struct pci_dev *dev, const char *name)
{
	unsigned char t;
	unsigned int u;
	int i;

/*
 * Find out what AMD IDE is this.
 */

	for (amd_config = amd_ide_chips; amd_config->id; amd_config++) {
			pci_read_config_byte(dev, PCI_REVISION_ID, &t);
			if (dev->device == amd_config->id && t >= amd_config->rev)
				break;
		}

	if (!amd_config->id) {
		printk(KERN_WARNING "AMD7409: Unknown AMD IDE Chip, contact Vojtech Pavlik <vojtech@suse.cz>\n");
		return -ENODEV;
	}

/*
 * Check UDMA66 mode or cable info set by BIOS.
 */

	switch (amd_config->flags & AMD_UDMA) {

		case AMD_UDMA_100:
			pci_read_config_byte(dev, AMD_CABLE_DETECT, &t);
			amd_80w = ((u & 0x3) ? 1 : 0) | ((u & 0xc) ? 2 : 0);
			break;

		case AMD_UDMA_66:
			pci_read_config_dword(dev, AMD_UDMA_TIMING, &u);
			for (i = 24; i >= 0; i -= 8)
				if ((u >> i) & 4)
					amd_80w |= (1 << (1 - (i >> 4)));
			break;
	}

	pci_read_config_dword(dev, AMD_IDE_ENABLE, &u);
	amd_enabled = ((u & 1) ? 2 : 0) | ((u & 2) ? 1 : 0);

/*
 * Disable prefetch & postwrite - AMD says it can crash the chip.
 */

	pci_read_config_byte(dev, AMD_IDE_CONFIG, &t);
	pci_write_config_byte(dev, AMD_IDE_CONFIG, t & 0x0f);

/*
 * Determine system bus clock.
 */

	amd_clock = system_bus_clock();
	if (amd_clock < 20 || amd_clock > 50) {
		printk(KERN_WARNING "AMD7409: User given PCI clock speed impossible (%d), using 33 MHz instead.\n", amd_clock);
		printk(KERN_WARNING "AMD7409: Use ide0=ata66 if you want to force UDMA66/UDMA100.\n");
		amd_clock = 33;
	}

/*
 * Print the boot message.
 */

	printk(KERN_INFO "AMD7409: AMD-%s IDE %s controller on pci%d:%d.%d\n",
			amd_config->name,
			(amd_config->flags & AMD_UDMA) == AMD_UDMA_100 ? "UDMA100" :
			(amd_config->flags & AMD_UDMA) == AMD_UDMA_66 ? "UDMA66" : "UDMA33",
			dev->bus->number, PCI_SLOT(dev->devfn), PCI_FUNC(dev->devfn));

/*
 * Register /proc/ide/via entry
 */

#ifdef CONFIG_PROC_FS
	if (!amd7409_proc) {
		for (i = 0; i < 6; i++)
                	if (pci_resource_flags(dev, i) & IORESOURCE_IO)
                        	break;

		amd_base = pci_resource_start(dev, i);
		bmide_dev = dev;
		amd7409_display_info = &amd_get_info;
		amd7409_proc = 1;

	}
#endif

	return 0;
}

unsigned int __init ata66_amd7409(ide_hwif_t *hwif)
{
	return ((amd_enabled & amd_80w) >> hwif->channel) & 1;
}

void __init ide_init_amd7409(ide_hwif_t *hwif)
{
	int i;

	hwif->tuneproc = &amd7409_tune_drive;
	hwif->speedproc = &amd_set_drive;
	hwif->autodma = 0;

	for (i = 0; i < 2; i++) {
		hwif->drives[i].io_32bit = 1;
		hwif->drives[i].unmask = 1;
		hwif->drives[i].autotune = 1;
		hwif->drives[i].dn = hwif->channel * 2 + i;
	}

#ifdef CONFIG_BLK_DEV_IDEDMA
	if (hwif->dma_base) {
		hwif->dmaproc = &amd7409_dmaproc;
		hwif->autodma = 1;
	}
#endif /* CONFIG_BLK_DEV_IDEDMA */
}

/*
 * We allow the BM-DMA driver only work on enabled interfaces.
 */

void __init ide_dmacapable_amd7409(ide_hwif_t *hwif, unsigned long dmabase)
{
	if ((amd_enabled >> hwif->channel) & 1)
		ide_setup_dma(hwif, dmabase, 8);
}
