#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <limits.h>
#include <time.h>

typedef uint32_t u32;
typedef uint64_t u64;

#define GOLDEN_RATIO_32 0x61C88647
#define GOLDEN_RATIO_64 0x61C8864680B583EBull

#define __hash_32 __hash_32_generic

static inline u32 __hash_32_generic(u32 val)
{
	return val * GOLDEN_RATIO_32;
}

static __always_inline u32 hash_64_generic(u64 val, unsigned int bits)
{
	/* 64x64-bit multiply is efficient on all 64-bit processors */
	return val * GOLDEN_RATIO_64 >> (64 - bits);
}
#define hash_64 hash_64_generic
#define hash_long(val, bits) hash_64(val, bits)
#define GOLDEN_RATIO_PRIME GOLDEN_RATIO_64

#define init_name_hash(salt)		(unsigned long)(salt)

static inline unsigned long
partial_name_hash(unsigned long c, unsigned long prevhash)
{
	return (prevhash + (c << 4) + (c >> 4)) * 11;
}

/*
The hashing function in end_name_hash() is replaced at v4.17-rc3 from
the 32-bit version to the 64-bit one.

  <linux/stringhash.h>: fix end_name_hash() for 64bit long
  https://github.com/torvalds/linux/commit/19b9ad67310ed2f685062a00aec602bec33835f0
	
  # git describe 19b9ad67310ed2f685062a00aec602bec33835f0
  v4.17-rc2-393-g19b9ad6
*/
static inline unsigned int end_name_hash(unsigned long hash)
{
	//return __hash_32((unsigned int)hash); // before v4.17-rc3
	return hash_long(hash, 32); // after v4.17-rc3
	//return (unsigned int)hash; // RHEL7.5 version
}

/* upstream version */

static unsigned int kernfs_name_hash(const char *name, const void *ns)
{
	unsigned long hash = init_name_hash(ns);
	unsigned int len = strlen(name);
	while (len--)
		hash = partial_name_hash(*name++, hash);
	hash = end_name_hash(hash);
	hash &= 0x7fffffffU;
	/* Reserve hash numbers 0, 1 and INT_MAX for magic directory entries */
	if (hash < 2)
		hash += 2;
	if (hash >= INT_MAX)
		hash = INT_MAX - 1;
	return hash;
}

#if 0
/* RHEL7.5 version */
static inline unsigned long hash_ptr(const void *ptr, unsigned int bits)
{
        return hash_long((unsigned long)ptr, bits);
}

static unsigned int kernfs_name_hash(const char *name, const void *ns)
{
        unsigned long hash = init_name_hash(ns);
        unsigned int len = strlen(name);
        while (len--)
                hash = partial_name_hash(*name++, hash);
        hash = (end_name_hash(hash) ^ hash_ptr((void *)ns, 31));
        hash &= 0x7fffffffU;
        /* Reserve hash numbers 0, 1 and INT_MAX for magic directory entries */
        if (hash < 2)
                hash += 2;
        if (hash >= INT_MAX)
                hash = INT_MAX - 1;
        return hash;
}
#endif

static void random_symbol(char *buf, size_t len)
{
	static char letters[] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_";

	buf[len] = '\0';
	while (len--)
		buf[len] = letters[rand() % (sizeof(letters) - 1)];
}

struct entry {
	unsigned int hash;
	char symbol[10];
};

static int entry_compare(const void *a, const void *b)
{
	const struct entry *p = a;
	const struct entry *q = b;

	return p->hash == q->hash ? 0
		: p->hash < q->hash ? 1
		: -1;
}

static int find_symbols_conflicting_hash(void)
{
	const unsigned long base_entry_size = 1UL << 20;
	struct entry *base;
	unsigned int i;
	static int srand_done;

	base = malloc(base_entry_size * sizeof(struct entry));
	if (!base) {
		perror("malloc");
		return -1;
	}

	if (!srand_done) {
		srand(time(NULL));
		srand_done = 1;
	}
	for (i = 0; i < base_entry_size; ++i) {
		struct entry *e = &base[i];

		random_symbol(e->symbol, 10);
		e->hash = kernfs_name_hash(e->symbol, NULL);
	}

	qsort(base, base_entry_size, sizeof(struct entry), entry_compare);

	for (i = 0; i < base_entry_size; ++i) {
		char symbol[10];
		unsigned int hash;
		unsigned int j;

		random_symbol(symbol, 10);
		hash = kernfs_name_hash(symbol, NULL);
		for (j = 0; j < base_entry_size; ++j) {
			if (base[j].hash < hash)
				break;
			if (base[j].hash == hash) {
				printf("%s %s %08x\n", symbol, base[j].symbol, hash);
				free(base);
				return 1;
			}
		}
	}

	free(base);
	return 0;
}

int main(int argc, char **argv)
{
	if (argc > 1) {
		int i;
		for (i = 1; i < argc; ++i) {
			char *symbol = argv[i];
			unsigned int hash = kernfs_name_hash(symbol, NULL);

			printf("%s 0x%08x\n", symbol, hash);
		}
	} else {
		int r;
		do {
			r = find_symbols_conflicting_hash();
			if (r < 0)
				break;
		} while (!r);
	}
			
	return 0;
}
