/*
 * Copyright 2013 Red Hat Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Authors: Jérôme Glisse <jglisse@redhat.com>
 */
/*
 * This test case check that we can migrate anonymous memory to device memory
 * and read from it.
 */
#include "hmm_test_framework.h"
#include <pthread.h>
#include <string.h>

#define CONCURRENT_MIGRATE_AND_READ

#define INITIAL_MIGRATE_THREADS 2
#define INITIAL_READ_THREADS 2

#define MAX_MIGRATE_THREADS 4
#define MAX_READ_THREADS 4

#define NRETRIES_INNER 100
#define NRETRIES 10
#define NPAGES 200
#define PAGE_SIZE (4 * 1024)
#define BUFFER_SIZE (NPAGES * PAGE_SIZE)

struct thread_context {
   struct hmm_buffer *buffer;
   struct hmm_ctx *ctx;
};

static void *migrate_thread_func(void *arg)
{
    struct thread_context *thread_context;
    struct stats s;
    long ret;
    int i;

    thread_context = (struct thread_context *)arg;

    for (i = 0; i < NRETRIES_INNER; i++) {
        /* Migrate buffer to remote memory. */
        ret = hmm_buffer_mirror_migrate_to(thread_context->ctx, thread_context->buffer, &s);

        if (ret != 0) {
            fprintf(stderr, "(EE:%d) hmm_buffer_mirror_migrate_to error %ld\n",
                    __LINE__, ret);
            ret = -1;
            goto out;
        }
    }

out:
    return (void*)ret;
}

static void *read_thread_func(void *arg)
{
    struct thread_context *thread_context;
    struct stats s;
    long ret;
    int i;

    thread_context = (struct thread_context *)arg;

    for (i = 0; i < NRETRIES_INNER; i++) {
        /* Read buffer to its mirror using dummy driver. */
        ret = hmm_buffer_mirror_read(thread_context->ctx, thread_context->buffer, 0, 
            thread_context->buffer->npages, &s);

        if (ret != 0) {
            fprintf(stderr, "(EE:%d) hmm_buffer_mirror_read error %ld\n",
                    __LINE__, ret);
            ret = -1;
            goto out;
        }
    }
    
out:
    return (void*)ret;
}

static int hmm_test(struct hmm_ctx *ctx, int num_migrate_threads, int num_read_threads)
{
    pthread_t migrate_threads[MAX_MIGRATE_THREADS], read_threads[MAX_READ_THREADS];
    struct thread_context thread_context;
    struct hmm_buffer *buffer;
    unsigned long i, size, k;
    int *ptr, ret = 0;

    fprintf(stdout, "&&& %d migrate threads, %d read threads: STARTING\n", num_migrate_threads, num_read_threads);

    HMM_BUFFER_NEW_ANON(buffer, BUFFER_SIZE);
    size = hmm_buffer_nbytes(buffer);

    for (k = 0; k < NRETRIES; k++) {
        thread_context.buffer = buffer;
        thread_context.ctx = ctx;

        /* Initialize buffer. */
        for (i = 0, ptr = buffer->ptr; i < size/sizeof(int); ++i) {
            ptr[i] = i + k;
        }

        for (i = 0; i < num_migrate_threads; i++) {
            if(pthread_create(&migrate_threads[i], NULL, migrate_thread_func, &thread_context))
                abort();
	    }

        for (i = 0; i < num_read_threads; i++) {
            if(pthread_create(&read_threads[i], NULL, read_thread_func, &thread_context))
                abort();
    	}

        for (i = 0; i < num_migrate_threads; i++) {
            pthread_join(migrate_threads[i], NULL);
        }

        for (i = 0; i < num_read_threads; i++) {
            pthread_join(read_threads[i], NULL);
        }

        if (k > 0 && k % 10000 == 0) {
            fprintf(stdout, "iteration %lu done\n", k);
        }
    }

    fprintf(stdout, "&&& %d migrate threads, %d read threads: PASSED\n", num_migrate_threads, num_read_threads);

    hmm_buffer_free(buffer);
    return ret;
}

int main(int argc, const char *argv[])
{
    struct hmm_ctx _ctx = {
        .test_name = "anon migration read test"
    };
    struct hmm_ctx *ctx = &_ctx;
    int num_migrate_threads;
    int num_read_threads;
    int ret = 0;

    for (num_migrate_threads = INITIAL_MIGRATE_THREADS; num_migrate_threads <= MAX_MIGRATE_THREADS; num_migrate_threads++) {
        for (num_read_threads = INITIAL_READ_THREADS; num_read_threads <= MAX_READ_THREADS; num_read_threads++) {
            ret = hmm_ctx_init(ctx);
            if (ret) {
                printf("!!! hmm_ctx_init FAILED WITH %d migrate threads, %d read threads\n", num_migrate_threads, num_read_threads);
                goto out;
            }

            ret = hmm_test(ctx, num_migrate_threads, num_read_threads);
            if (ret) {
                printf("!!! hmm_test FAILED WITH %d migrate threads, %d read threads\n", num_migrate_threads, num_read_threads);
                goto out;
            }
        
            hmm_ctx_fini(ctx);
        }
    }

out:
    fprintf(stderr, "(%s)[%s] %s\n", ret ? "EE" : "OK", argv[0], ctx->test_name);
    return ret;
}
