#ifndef _QSPINLOCK_H
#define _QSPINLOCK_H

//#define CONFIG_NR_CPUS	(1U << 14)
#define CONFIG_NR_CPUS		256

#include "atomic.h"

/*
 * The queue spinlock data structure - a 32-bit word
 * Bits 0-7 : Set if locked
 * Bits 8-31: Queue code
 */
typedef struct qspinlock {
	atomic_t	val;
} arch_spinlock_t;

#define _Q_LOCKED_OFFSET	0
#define _Q_LOCKED_BITS		8
#define _Q_LOCKED_MASK		(((1U << _Q_LOCKED_BITS) - 1) << _Q_LOCKED_OFFSET)

#define _Q_PENDING_OFFSET	(_Q_LOCKED_OFFSET + _Q_LOCKED_BITS)
#if CONFIG_NR_CPUS < (1U << 14)
#define _Q_PENDING_BITS		8
#define VERSION "qspinlock8"
#else
#define _Q_PENDING_BITS		1
#define VERSION "qspinlock1"
#endif
#define _Q_PENDING_MASK		(((1U << _Q_PENDING_BITS) - 1) << _Q_PENDING_OFFSET)

#define _Q_TAIL_IDX_OFFSET	(_Q_PENDING_OFFSET + _Q_PENDING_BITS)
#define _Q_TAIL_IDX_BITS	2
#define _Q_TAIL_IDX_MASK	(((1U << _Q_TAIL_IDX_BITS) - 1) << _Q_TAIL_IDX_OFFSET)

#define _Q_TAIL_CPU_OFFSET	(_Q_TAIL_IDX_OFFSET + _Q_TAIL_IDX_BITS)
#define _Q_TAIL_CPU_BITS	(32 - _Q_TAIL_CPU_OFFSET)
#define _Q_TAIL_CPU_MASK	(((1U << _Q_TAIL_CPU_BITS) - 1) << _Q_TAIL_CPU_OFFSET)

#define _Q_TAIL_OFFSET		_Q_TAIL_IDX_OFFSET
#define _Q_TAIL_MASK		(_Q_TAIL_IDX_MASK | _Q_TAIL_CPU_MASK)

#define _Q_LOCKED_VAL		(1U << _Q_LOCKED_OFFSET)
#define _Q_PENDING_VAL		(1U << _Q_PENDING_OFFSET)



/*
 * External function declarations
 */
extern void queue_spin_lock_slowpath(struct qspinlock *lock, u32 val);

/**
 * queue_spin_is_locked - is the spinlock locked?
 * @lock: Pointer to queue spinlock structure
 * Return: 1 if it is locked, 0 otherwise
 */
static __always_inline int queue_spin_is_locked(struct qspinlock *lock)
{
	return atomic_read(&lock->val) & _Q_LOCKED_MASK;
}

/**
 * queue_spin_value_unlocked - is the spinlock structure unlocked?
 * @lock: queue spinlock structure
 * Return: 1 if it is unlocked, 0 otherwise
 */
static __always_inline int queue_spin_value_unlocked(struct qspinlock lock)
{
	return !queue_spin_is_locked(&lock);
}

/**
 * queue_spin_is_contended - check if the lock is contended
 * @lock : Pointer to queue spinlock structure
 * Return: 1 if lock contended, 0 otherwise
 */
static __always_inline int queue_spin_is_contended(struct qspinlock *lock)
{
	return atomic_read(&lock->val) >> _Q_LOCKED_OFFSET;
}
/**
 * queue_spin_trylock - try to acquire the queue spinlock
 * @lock : Pointer to queue spinlock structure
 * Return: 1 if lock acquired, 0 if failed
 */
static __always_inline int queue_spin_trylock(struct qspinlock *lock)
{
	return !atomic_read(&lock->val) &&
	       atomic_cmpxchg(&lock->val, 0, _Q_LOCKED_VAL) == 0;
}

/**
 * queue_spin_lock - acquire a queue spinlock
 * @lock: Pointer to queue spinlock structure
 */
static __always_inline void queue_spin_lock(struct qspinlock *lock)
{
	u32 val = atomic_cmpxchg(&lock->val, 0, _Q_LOCKED_VAL);

	if (likely(val == 0))
		return;

	queue_spin_lock_slowpath(lock, val);
}

/**
 * queue_spin_unlock - release a queue spinlock
 * @lock : Pointer to queue spinlock structure
 */
#ifndef queue_spin_unlock
static __always_inline void queue_spin_unlock(struct qspinlock *lock)
{
#if 0
	smp_mb__before_atomic_dec();
	atomic_sub(_Q_LOCKED_VAL, &lock->val);
#else
	smp_store_release((u8 *)lock, 0);
#endif
}
#endif

/*
 * Initializier
 */
#define	__ARCH_SPIN_LOCK_UNLOCKED	{ .val = ATOMIC_INIT(0), }

/*
 * Remapping spinlock architecture specific functions to the corresponding
 * queue spinlock functions.
 */
#define arch_spin_is_locked(l)		queue_spin_is_locked(l)
#define arch_spin_is_contended(l)	queue_spin_is_contended(l)
#define arch_spin_value_unlocked(l)	queue_spin_value_unlocked(l)
#define arch_spin_lock(l)		queue_spin_lock(l)
#define arch_spin_trylock(l)		queue_spin_trylock(l)
#define arch_spin_unlock(l)		queue_spin_unlock(l)
#define arch_spin_lock_flags(l, f)	queue_spin_lock(l)

#endif /* _QSPINLOCK_H */
