#!/bin/sh

### LAST MODIFIED: 07-JAN-2015 (Sedat Dilek <sedat.dilek@gmail.com>)

### HELP
# 1. make deb-pkg (options):  <file: scripts/package/builddeb>
# 2. localversion handling:   <file: scripts/setlocalversion>
# 3. KBUILD_USER_* variables: <file: scripts/mkcompile_h>
# 4. Prereqs/Preparation:     <http://kernel-handbook.alioth.debian.org/ch-common-tasks.html#s4.2.1>

### PREREQS: Install build-essential, fakeroot, dpkg-dev and git packages
#apt-get update
#apt-get install build-essential fakeroot dpkg-dev git

### Language settings
export LANG=C
export LC_ALL=C

### Base is Linux-upstream source directory
BASE_DIR=$(pwd)
SRC_DIR="${BASE_DIR}/linux"

### Clone Linux-upstream remote GIT tree
#GIT_URL="git://git.kernel.org/pub/scm/linux/kernel/git/torvalds/linux.git"
#git clone ${GIT_URL}

### Change to build directory
cd ${SRC_DIR}

### Make and compiler settings (here: Use mycompiler wrapper-script suggested by Linus Torvalds)
## XXX: DEBUG: MAKE="make V=1" (verbose output to get $mycompiler lines)
## XXX: DEBUG: MAKE_JOBS="1"
## NOTE: For fast builds use MAKE_JOBS=$(($(getconf _NPROCESSORS_ONLN)+1))
MAKE="make V=1"
MAKE_JOBS="2"
CC_FOR_BUILD="mycompiler"
HOSTCC_FOR_BUILD="$CC_FOR_BUILD"

### Extract version-tag from local GIT tree
LATEST_UPSTREAM_VER_TAG=$(git for-each-ref --sort=taggerdate --format="%(refname:short)" refs/tags | grep ^'v[0-9].[0-9]*' | tail -1)

### Distribution settings
# NOTE: $distribution is used in $debian_revision
distribution=$(lsb_release --codename --short)

### Uploader settings
# NOTE: $uploader is used in $debian_revision
export DEBFULLNAME="Sedat Dilek"
export DEBEMAIL="sedat.dilek@gmail.com"
uploader="dileks"

### Debian upstream version settings 
# NOTE-1: $upstream_version is extracted from Makefile
# NOTE-2: $debian_upstream_version is used in builddeb options
# NOTE-3: Convert '-rcX' in $upstream_version to '~rcX' (see $extraversion)
# EXAMPLE: debian_upstream_version="3.19.0~rc2"
version=$(awk '/^VERSION = / {print $3}' Makefile)
patchlevel=$(awk '/^PATCHLEVEL = / {print $3}' Makefile)
sublevel=$(awk '/^SUBLEVEL = / {print $3}' Makefile)
extraversion=$(awk '/^EXTRAVERSION = / {print $3}' Makefile)
upstream_version="${version}.${patchlevel}.${sublevel}${extraversion}"
debian_upstream_version=$(echo $upstream_version | tr '-' '~')

### Debian revision settings
# NOTE-1: $debian_revision is used in builddeb options
# NOTE-2: $revision is used in $debian_revision and $mylocalversion
# EXAMPLE: debian_revision="1~precise+dileks1"
revision="1"
uploader_revision="1"
debian_revision="${revision}~${distribution}+${uploader}${uploader_revision}"

### My kernel-featureset settings
# EXAMPLE-1: Use own "iniza" featureset (default)
# EXAMPLE-2: Try "llvmlinux" featureset (EXPERIMENTAL)
myfeatureset="iniza"

### My kernel-flavour settings
# EXAMPLE-1: Use own "small" kernel-flavour (localmodconfig-ed plus some debug-options enabled) 
# EXAMPLE-2: Use own "debug" kernel-flavour (CONFIG_DEBUG_INFO=y, WARNING: Enough disc-space?)
mykernelflavour="small"

### Customized LOCALVERSION settings
# NOTE-1: $mylocalversion is attached as suffix to "include/generated/utsrelease.h" file
# NOTE-2: Usage of LOCALVERSION=$mylocalversion suppresses "+" suffix
# NOTE-3: See also "#define UTS_RELEASE" in utsrelease.h file.
# WARNING: Do NOT use underscore ("_") in $mylocalversion (see <http://bugs.debian.org/222467>).
# ERR-MSG: dpkg-gencontrol: error: Illegal package name ...
# EXAMPLE-1: mylocalversion="1-iniza-small"
# EXAMPLE-2: mylocalversion="1-iniza-x86_64" ("x86_64" not allowed, use "amd64" instead)
mylocalversion="${revision}-${myfeatureset}-${mykernelflavour}"

### Build-log file
KVER="${upstream_version}-${mylocalversion}"
BUILD_LOG_FILE="build-log_${KVER}.txt"

### make options
MAKE_OPTS="CC=${CC_FOR_BUILD} HOSTCC=$HOSTCC_FOR_BUILD -j${MAKE_JOBS} KBUILD_BUILD_USER=${DEBEMAIL} LOCALVERSION=-${mylocalversion}"

### builddeb (deb-pkg) options
# EXAMPLE-1: BUILDDEB_TARGET="deb-dpkg"    (see scripts/package/builddeb)
# EXAMPLE-2: BUILDDEB_TARGET="bindeb-pkg"  (generate binary kernel package only, Linux >= v4.3-rc1)
# EXAMPLE-3: KDEB_PKGVERSION="3.19.0~rc2-1~precise+dileks1"
# EXAMPLE-4: KDEB_CHANGELOG_DIST="precise" (set distribution name explicitly)
BUILDDEB_TARGET="bindeb-pkg"
BUILDDEB_OPTS="KDEB_PKGVERSION=${debian_upstream_version}-${debian_revision}"
##BUILDDEB_OPTS="$BUILDDEB_OPTS KDEB_CHANGELOG_DIST=$distribution"

### List some version and option settings
echo "Linux upstream version .... ${LATEST_UPSTREAM_VER_TAG}"
echo "Debian upstream version ... ${debian_upstream_version}"
echo "Debian revision ........... ${debian_revision}"
echo "make options .............. ${MAKE_OPTS}"
echo "builddeb target ........... ${BUILDDEB_TARGET}"
echo "builddeb options .......... ${BUILDDEB_OPTS}"
echo ""

### Remove any existing localversion* file(s) as customized LOCALVERSION is used
# EXAMPLE: Linux-next ships a "localversion-next" file.
rm -f localversion*

### Start build
echo "##### Starting Linux-kernel build ..."
fakeroot ${MAKE} ${MAKE_OPTS} ${BUILDDEB_TARGET} ${BUILDDEB_OPTS} 2>&1 | tee ../${BUILD_LOG_FILE}
echo "##### Finished!"
