#include <linux/init.h>
#include <linux/module.h>

#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/bigphysarea.h>

/* name is different in 2.0.x and 2.1.x */
#if LINUX_VERSION_CODE < 0x020100
#define ioremap vremap
#define iounmap vfree
#endif


/* nice and high */
static int major = 42;
static int first = 1;

/*
 * Our special open code.
 * MOD_INC_USE_COUNT make sure that the driver memory is not freed
 * while the device is in use.
 */
static int mod_open( struct inode* ino, struct file* filep)
{
	if (first) {
		first = 0;
	}
	return -1;
}

/*
 * Now decrement the use count.
 */
static int mod_release( struct inode* ino, struct file* filep)
{
	first = 1;
	return -1;
}

static struct file_operations fops = {
	open: mod_open,
	release: mod_release
};

#define COUNT 10

static void *addr[COUNT];
static caddr_t buff;

void test_bigphysarea_alloc(void)
{
	int i;
	unsigned long sizebuff=bigphysarea_get_size()-(10*1024*1024);

	for (i = 0; i < COUNT; i++) {
		addr[i] = bigphysarea_alloc_pages(3, 2, GFP_KERNEL);
		printk("addr[%d] = 0x%08lx\n", i, (unsigned long)addr[i]);
	}
	
	buff=bigphysarea_alloc(sizebuff);
	if (buff == NULL)
	{
		printk("Failed to allocate a %ld MB buffer\n", sizebuff/(1024*1024));
	}
	else
	{
		for (i = 0; i < sizebuff; i++)
		{
			buff[i]=i;
			if (i % (10*1024*1024) == 0)
			{
				printk("%d bytes of the bigphysarea pool touched\n",i);
			}
		}
	}		
}

void test_bigphysarea_free(void)
{
	int i;

	for (i = 0; i < COUNT; i++) {
		bigphysarea_free_pages(addr[i]);
	}
	if (buff != NULL)
	{
		bigphysarea_free(buff, 0);
	}
}


/*
 * And now the modules code and kernel interface.
 */
int __init test_bigphysarea_init(void)
{
	if (register_chrdev(major, "test-module", &fops)) {
		printk("register_chrdev failed: goodbye world :-(\n");
		return -EIO;
	}

	printk("test-module loaded\n");

	test_bigphysarea_alloc();

	return 0;
}

void __exit test_bigphysarea_exit(void)
{

	test_bigphysarea_free();

	if (unregister_chrdev(major, "test-module") != 0)
		printk("cleanup_module failed\n");
	else
		printk("test-module removed\n");
}

module_init(test_bigphysarea_init);
module_exit(test_bigphysarea_exit);
