/*
 * 2007+ Copyright (c) Evgeniy Polyakov <johnpol@2ka.mipt.ru>
 * All rights reserved.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/resource.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>

#include "ntl.h"
#include "list.h"
#include "lock.h"
#include "ntl_kevent.h"
#include "ntl_user.h"
#include "ntl_sched.h"

static struct ntl_lock thread_cache_lock;
static LIST_HEAD(thread_cache_list);
static int thread_cache_limit = 40*1024*1024; /* 40 Mb maximum thread cache size */
static int thread_cache_size;

static int ntl_created, ntl_removed;

/* Default stack size obtained from rlimits */
static int thread_default_size;

static void ntl_thread_free(struct ntl_thread *th)
{
	struct ntl_thread *s;
	int remove = 0;

	ntl_lock(&thread_cache_lock);
	if (thread_cache_size < thread_cache_limit) {
		int added = 0;
		thread_cache_size += th->stack_size;
		list_for_each_entry(s, &thread_cache_list, thread_cache_entry) {
			if (s->stack_size >= th->stack_size) {
				list_add(&th->thread_cache_entry, s->thread_cache_entry.prev);
				added = 1;
				break;
			}
		}

		if (!added)
			list_add_tail(&th->thread_cache_entry, &thread_cache_list);
	} else
		remove = 1;
	ntl_unlock(&thread_cache_lock);
	if (remove)
		munmap(NULL, th->stack_size + PAGE_SIZE + ALIGN(sizeof(struct ntl_thread), PAGE_SIZE));
}

static struct ntl_thread *ntl_thread_alloc(int osize)
{
	struct ntl_thread *s, *ret = NULL;
	int size;

	ntl_lock(&thread_cache_lock);
	list_for_each_entry(s, &thread_cache_list, thread_cache_entry) {
		if (s->stack_size >= osize) {
			ret = s;
			break;
		}
	}

	if (ret && ((ret->stack_size>>2) < osize)) {
		thread_cache_size -= ret->stack_size;
		list_del(&ret->thread_cache_entry);
		ntl_unlock(&thread_cache_lock);
		return ret;
	}
	ntl_unlock(&thread_cache_lock);

	size = osize + PAGE_SIZE + ALIGN(sizeof(struct ntl_thread), PAGE_SIZE);

	ret = mmap(NULL, size, PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_ANONYMOUS, -1, 0);
	if (ret == MAP_FAILED)
		return NULL;

	ret->stack_size = osize;
	ret->guard_page = (void *)ret + ALIGN(sizeof(struct ntl_thread), PAGE_SIZE);
	ret->stack = ret->guard_page + PAGE_SIZE;
	
	ulog("%s: mmap, stack: %p, size: %d.\n", __func__, ret->stack, ret->stack_size);

	return ret;
}

int  __attribute__ ((constructor)) ntl_thread_sysinit(void)
{
	struct rlimit rl;
	int err;

	err = getrlimit (RLIMIT_STACK, &rl);
	if (err)
		exit(err);
	thread_default_size = rl.rlim_cur;

	ntl_lock_init(&thread_cache_lock);

	err = ntl_sched_init();
	if (err)
		exit(err);

	err =  ntl_kevent_init();
	if (err)
		exit(err);

	if (1) {
		struct ntl_thread_id this_id = {.id = -1,};
		err =  ntl_thread_create(&this_id, NULL, NULL);
	}


	return 0;
}

int  __attribute__ ((destructor)) ntl_thread_sysexit(void)
{
	ntl_sched_exit();
	return 0;
}

void ntl_thread_exit(struct ntl_thread *th)
{
	ntl_sched_remove(th);
	ntl_thread_free(th);
	ntl_removed++;
}

int ntl_thread_create(struct ntl_thread_id *id, void *(*func)(void *), void *data)
{
	struct ntl_thread *th;
	int err;

	th = ntl_thread_alloc(thread_default_size);
	if (!th) {
		ulog("%s: failed to allocate a thread.\n", __func__);
		return -ENOMEM;
	}
	ulog("%s: new th: %p.\n", __func__, th);

	th->priv = data;
	th->func = func;
	th->last = 0;
	th->prio = 100;

	err = ntl_sched_add(th);
	if (err) {
		ulog("%s: failed to register thread in scheduler, err: %d.\n", __func__, err);
		goto err_out_free;
	}

	id->id = th->tid;

	ntl_created++;

	ulog("created: %d, removed: %d.\n", ntl_created, ntl_removed);

	ntl_schedule();

	return 0;

err_out_free:
	ntl_thread_free(th);
	ulog("%s: exiting with error %d [%x].\n", __func__, err, err);
	return err;
}
