/*
 * 2007+ Copyright (c) Evgeniy Polyakov <johnpol@2ka.mipt.ru>
 * All rights reserved.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __I386_LOCK_H
#define __I386_LOCK_H

#include <signal.h>

typedef struct { volatile int counter; } atomic_t;
#define ADDR (*(volatile long *) addr)

struct ntl_lock
{
	unsigned long		__lock;
};

static inline void atomic_inc(atomic_t *v)
{
	asm volatile (
		"lock; incl %0\n"
		:"=m"(v->counter)
		:"m"(v->counter)
	);
}

static inline void atomic_dec(atomic_t *v)
{
	asm volatile (
		"lock; decl %0\n"
		:"=m"(v->counter)
		:"m"(v->counter)
	);
}

static inline int atomic_dec_and_test(atomic_t *v)
{
	unsigned char c;

	asm volatile (
		"lock; decl %0; sete %1\n"
		:"+m" (v->counter), "=qm" (c)
		: : "memory");
	return c != 0;
}

static inline void set_bit(int nr, volatile unsigned long * addr)
{
	asm volatile( 
		"lock; btsl %1,%0"
		:"+m" (ADDR)
		:"Ir" (nr));
}

static inline void clear_bit(int nr, volatile unsigned long * addr)
{
	asm volatile(
		"lock; btrl %1,%0\n"
		:"+m" (ADDR)
		:"Ir" (nr));
}

static inline int test_and_set_bit(int nr, volatile unsigned long * addr)
{
	int oldbit;

	asm volatile(
		"lock; btsl %2,%1; sbbl %0,%0\n"
		:"=r" (oldbit),"+m" (ADDR)
		:"Ir" (nr) : "memory");
	return oldbit;
}

static inline int test_and_clear_bit(int nr, volatile unsigned long * addr)
{
	int oldbit;

	asm volatile ( 
		"lock; btrl %2,%1\n\tsbbl %0,%0"
		:"=r" (oldbit),"+m" (ADDR)
		:"Ir" (nr) : "memory");
	return oldbit;
}

static inline void __ntl_lock(struct ntl_lock *lock)
{
	while (test_and_set_bit(0, &lock->__lock));

}

static inline void __ntl_unlock(struct ntl_lock *lock)
{
	clear_bit(0, &lock->__lock);
}

static inline void ntl_lock(struct ntl_lock *lock)
{
	sigset_t s;

	sigemptyset(&s);
	sigaddset(&s, SIGALRM);
	sigprocmask(SIG_BLOCK, &s, NULL);
	__ntl_lock(lock);

}

static inline void ntl_unlock(struct ntl_lock *lock)
{
	__ntl_unlock(lock);
	sigset_t s;

	sigemptyset(&s);
	sigaddset(&s, SIGALRM);
	sigprocmask(SIG_UNBLOCK, &s, NULL);
}

static inline void ntl_lock_init(struct ntl_lock *lock)
{
	lock->__lock = 0;
}

#endif /* __I386_LOCK_H */
