/*
 * 2007+ Copyright (c) Evgeniy Polyakov <johnpol@2ka.mipt.ru>
 * All rights reserved.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>

#include "ntl.h"
#include "lock.h"
#include "ntl_sched.h"

register unsigned long get_ebp asm ("ebp");

int ntl_arch_call_func(void *(*func)(void *), void *data, void *stack, unsigned long stack_size)
{
	asm volatile (
		"pushl	%%ebx\n"
		"mov	%0,%%eax	\n" /* Start address */
		"mov	%1,%%ebx	\n" /*  Arguments */
		"mov	%%esp,%%edx	\n" /* save old sp to edx */
		"mov	%2,%%esp	\n" /* change stack */

		"push	%%edx		\n"
		"push	%%ebx		\n" /* copy arguments to new stack */

		"call	*%%eax		\n" /* call (*func)(arg); */
		"mov	4(%%esp),%%edx	\n"
		"mov	%%edx,%%esp\n" /* restore old stack */
		"popl	%%ebx\n"
		:
		: "g"(func), "g"(data), "g"(stack+stack_size)
		: "edx", "esp", "eax"
	);
	return 0;
}

static void ntl_arch_start(void)
{
	unsigned long reg;
	struct ntl_thread *th;

	asm volatile ("movl %%ebx, %0\n": "=rm"(reg));
	th = (struct ntl_thread *)reg;

	ntl_arch_call_func(th->func, th->priv, th->stack, th->stack_size);
	ntl_thread_exit(th);
	kill(getpid(), SIGALRM);
}

extern struct ntl_thread *th_setup;

void ntl_arch_sched_sigsetup(int signo __attribute__ ((unused)))
{
	struct sigframe *frame = (void *)get_ebp + sizeof(void *);
	struct __sigcontext *sc = &frame->sc;
	struct ntl_thread *th = th_setup;

	memcpy(&th->frame.sc, sc, sizeof(struct sigcontext));
	th->frame.sc.eip = (unsigned long)ntl_arch_start;
	th->frame.sc.ebx = (unsigned long)th;
	th->frame.sc.esp = (unsigned long)th->stack + th->stack_size/2;
	th->frame.sc.ebp = (unsigned long)th->stack + th->stack_size/2;
	set_bit(0, &th->flags);
	ulog("%s: setup: %p, data: %p, flags: %lx.\n", 
			__func__, th, th->priv, th->flags);
}

void ntl_arch_sched_sighandler(int signo __attribute__ ((unused)))
{
	struct sigframe *frame = (void *)get_ebp + sizeof(void *);
	struct __sigcontext *sc = &frame->sc;
	struct ntl_thread *new, *old;
	
	old = new = NULL;
	ntl_sched_find_current(&old, &new);

	ulog("%s: start %p [%lx] -> %p [%lx].\n", 
			__func__, 
			old, (old)?old->frame.sc.eip:0, 
			new, (new)?new->frame.sc.eip:0);
	if (old)
		memcpy(&old->frame.sc, sc, sizeof(struct sigcontext));
	if (new && new->frame.sc.eip != 0)
		memcpy(sc, &new->frame.sc, sizeof(struct sigcontext));

	ulog("%s: stop %p [%lx] -> %p [%lx].\n", 
			__func__, 
			old, (old)?old->frame.sc.eip:0, 
			new, (new)?new->frame.sc.eip:0);
}
