/*	SYN cookies firewall version 0.9.1 for Linux kernel 2.2.12 - 2.2.17.
	Written by Bordi Zhou(bordi@bronzesoft.org) Oct. 20, 2000.

	This is the use space command used to set the firewall rule.

	The SYN cookies firewall project is under the terms of GNU General
	Public License (GPL) as published by the Free Software Foundation.
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <asm/unaligned.h>
#include <asm/param.h>	/* For HZ. */

#include "ip_scfw.h"

void usage(char *str)
{
	printf( "%s"
			"ipscfwadm [-h] [-F] [-L] [-D rulenum] [{-A|-I rulenum} [-s address[/mask]]\n"
			"[-p port1[-port2]] [-f {host|port}] [-c maxcnt] [-t timeout]]\n"
			"Type ipscfwadm -h for help.\n",str);
}
int parse_server(struct sc_fw_rule *rule,char *arg)
{
	char buf[128], *p;
	int shift;

	if(arg[0]=='!')
	{	rule->invflg |= SC_FW_INV_ADDR;
		arg++;
	}
	strncpy(buf,arg,sizeof(buf)-1);
	if( (p=strchr(buf,'/'))!=NULL )
	{	*p = '\0';
		shift=atoi(p+1);
		if( shift>32 || shift==31 )	/* Not a valid mask. */
			return -1;
		rule->mask.s_addr=htonl(0xffffffff<<(32-shift));
	}
	else
		rule->mask.s_addr = 0xffffffff;
	if(!inet_aton(buf,&(rule->addr)))
		return -1;
	return 0;	/* Success. */
}
int parse_ports(struct sc_fw_rule *rule,char *arg)
{
	char buf[128], *p;
	if(arg[0]=='!')
	{	rule->invflg |= SC_FW_INV_PORT;
		arg++;
	}
	strncpy(buf,arg,sizeof(buf)-1);
	if( (p=strchr(buf,'-'))!=NULL )
	{	*p = '\0';
		rule->ports[1] = atoi(p+1);
	}
	else
		rule->ports[1] = atoi(buf);
	rule->ports[0] = atoi(buf);
	if( rule->ports[0] > rule->ports[1] )
		return -1;	/* Invalid range. */
	return 0;
}
int parse_field(struct sc_fw_rule *rule,char *arg)
{
	if(strcmp(arg,"host")==0)
		rule->field = FIELD_HOST;
	else if(strcmp(arg,"port")==0)
		rule->field = FIELD_PORT;
	else
		return -1;
	return 0;
}
void cmd_help(void)
{
	printf("ipscfwadm 1.0 Oct. 20, 2000\n\n");
	printf( "ipscfwadm [-h] [-F] [-L] [-D rulenum] [{-A|-I rulenum} [-s address[/mask]]\n"
			"[-p port1[-port2]] [-f {host|port}] [-c maxcnt] [-t timeout]]\n\n");
	printf("Options:\n");
	printf("  %-19s%-20s\n","-F","Flush rules.");
	printf("  %-19s%-20s\n","-L","List rules.");
	printf("  %-19s%-20s\n","-D rulenum","Delete a rule. Rulenum is the position of the rule.");
	printf("  %-19s%-20s\n","-A","Append a rule.Can be followed with -s,-p,-f,-c,-t.");
	printf("  %-19s%-20s\n","-I rulenum","Insert a rule. Rulnenum is the position of the rule.");
	printf("  %-19s%-20s\n"," ","Can be followed with -s,-p,-f,-c,-t.");
	printf("  %-19s%-20s\n","-s address[/mask]","Specify the protected server.");
	printf("  %-19s%-20s\n","-p port1[-port2]","Range of the protected ports.");
	printf("  %-19s%-20s\n","-f host|port","The field of the count.");
	printf("  %-19s%-20s\n","-c maxcnt","Maximal count of half-open connections.");
	printf("  %-19s%-20s\n","-t timeout","Timeout of half-open connections.");
	printf("Type man ipscfwadm for more help.\n");
}
void cmd_list_rules(void)
{
#define SCFW_PROC_FILE	"/proc/net/ip_fw_syncookies"
	FILE *fp;
	struct sc_fw_rule rule;
	int n=1,maskcnt;
	unsigned long mask;
	char address[128],ports[128];

	if( (fp=fopen(SCFW_PROC_FILE,"r"))==NULL )
		return;
	printf("%-3s   %-20s%-19s%-9s%-8s%s\n",
			"num","address","ports","field","maxcnt","timeout");
	while( fscanf(fp,
			"%X/%X "		/* Address. */
			"%hu-%hu "		/* Ports. */
			"%hu "			/* Invert flags. */
			"%d "			/* Field. */
			"%d "			/* Maxcnt of half opened connection. */
			"%d",			/* Timeout of a half opened connection. */
			&rule.addr.s_addr,
			&rule.mask.s_addr,
			&rule.ports[0], &rule.ports[1],
			&rule.invflg,
			&rule.field,
			&rule.maxcnt,
			&rule.timeout)==8 )
	{	maskcnt = 0;
		mask = rule.mask.s_addr;
		while(mask)
		{	maskcnt++;	/* Counts the mask bits. */
			mask = mask<<1;
		}
		rule.addr.s_addr = htonl(rule.addr.s_addr);
		sprintf(address,"%s%s/%d",((rule.invflg&SC_FW_INV_ADDR)?"!":""),
			inet_ntoa(rule.addr),maskcnt);
		sprintf(ports,"%s%u - %u",((rule.invflg&SC_FW_INV_PORT)?"!":""),
			rule.ports[0],rule.ports[1]);
		printf("%-3d:  %-20s%-19s"
			"%-9s%-8d%d\n",n++,address,
			ports,(rule.field==FIELD_HOST?"HOST":"PORT"),
			rule.maxcnt,rule.timeout/HZ);
	}
	fclose(fp);
}
static int do_setsockopt(int command, const void *data, int length)
{
	int sockfd;		/* The raw socket used to conctrl the sc_fw_rule. */
	if( (sockfd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW))<0 )
		return -1;
	if(setsockopt(sockfd, IPPROTO_IP, command, (char *)data, length)<0)
		return -1;
	close(sockfd);
	return 0;
}

void append_rule(struct sc_fw_rule *rule)
{
	if(do_setsockopt(IP_SCFW_APPEND,(void *)rule,sizeof(struct sc_fw_rule))<0)
		perror("Error append the rule.\n");
}
void insert_rule(struct sc_fw_rule *rule,int rulenum)
{
	struct scfw_insert_rule ins_rule = { rulenum, *rule };

	if(do_setsockopt(IP_SCFW_INSERT,(void *)&ins_rule, \
		sizeof(struct scfw_insert_rule))<0)
		perror("Error insert the rule.\n");
}
void cmd_delete_rule(int rulenum)
{
	if(do_setsockopt(IP_SCFW_DELETE,(void *)&rulenum,sizeof(int))<0)
		perror("Error delete the rule.\n");
}
void cmd_flush(void)
{
	char c;
	if(do_setsockopt(IP_SCFW_FLUSH,(void *)&c,0)<0)
		perror("Error flush the rule.\n");
}

int main(int argc, char **argv)
{
	char c;
	char cmd = 0;
	struct sc_fw_rule rule;
	int rulenum=0;
	rule.addr.s_addr = 0L;
	rule.mask.s_addr = 0L;
	rule.ports[0] = 0;
	rule.ports[1] = 65535;	/* Default ports range is from 0 to 65535. */
	rule.invflg = 0;
	rule.field = FIELD_PORT;
	rule.maxcnt = DEFAULT_MAXCNT;
	rule.timeout = DEFAULT_TIMEOUT;

	while( (c=getopt(argc,argv,"hLFAD:I:s:p:f:c:t:"))!=EOF )
	{	switch(c)
		{	case 'F':
				if(cmd!=0)
				{	usage("Commands conflict.\n");
					return -1;
				}
				cmd_flush();
				return 0;
			case 'L':
				if(cmd!=0)
				{	usage("Commands conflict.\n");
					return -1;
				}
				cmd_list_rules();
				return 0;
			case 'D':
				if(cmd!=0)
				{	usage("Commands conflict.\n");
					return -1;
				}
				if( (rulenum = atoi(optarg))<=0 )
				{	usage("Rule number must greater than 0.\n");
					return -1;
				}
				cmd_delete_rule(rulenum);
				return 0;
			case 'A':
				if(cmd!=0)
				{	usage("Commands conflict.\n");
					return -1;
				}
				cmd = c;
				break;
			case 'I':
				if(cmd!=0)
				{	usage("Commands conflict.\n");
					return -1;
				}
				if( (rulenum = atoi(optarg))<=0 )
				{	usage("Rule number must greater than 0.\n");
					return -1;
				}
				cmd = c;
				break;
			case 's':
				if(parse_server(&rule,optarg))
				{	usage("Bad server.\n");
					return -1;
				}
				break;
			case 'p':
				if(parse_ports(&rule,optarg))
				{	usage("Bad port.\n");
					return -1;
				}

				break;
			case 'f':
				if(parse_field(&rule,optarg))
				{	usage("Unknow field.\n");
					return -1;
				}
				break;
			case 'c':
				if( (rule.maxcnt = atoi(optarg))<=0 )
					rule.maxcnt = DEFAULT_MAXCNT;
				break;
			case 't':
				if( (rule.timeout = atoi(optarg)*HZ)<=0 )
					rule.timeout = DEFAULT_TIMEOUT*HZ;
				break;
			case 'h':
				cmd_help();
				return 0;
			default:
				usage("Unknow parameter.\n");
				return -1;
		}
	}
	switch(cmd)
	{	case 'A':
			append_rule(&rule);
			break;
		case 'I':
			insert_rule(&rule,rulenum);
			break;
		default:
			usage("");
			break;
	}
	return 0;
}
