/*
 * Gradient demo (using 1D texture)
 * Brian Paul
 * 21 April 2004
 */


#define GL_GLEXT_PROTOTYPES
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <GL/glut.h>

#define PI 3.14159

#define LINEAR 0
#define RADIAL 1
static int GradientMode = LINEAR;

#define CLAMP 0
#define REPEAT 1
static int WrapMode = CLAMP;

static GLint WinWidth = 500, WinHeight = 500;
static GLfloat Xpos, Ypos;


static void
PrintString(const char *s)
{
   while (*s) {
      glutBitmapCharacter(GLUT_BITMAP_8_BY_13, (int) *s);
      s++;
   }
}


/*
 * Compute radial texcoord from x, y
 */
static void
MyTexgen(float x, float y)
{
   float scale, dx, dy, r;
   scale = 1.5 / sqrt( 0.25 * WinWidth * WinWidth + 0.25 * WinHeight * WinHeight);
   dx = (x + Xpos) - 0.5 * WinWidth;
   dy = (y + Ypos) - 0.5 * WinHeight;
   r = sqrt(dx * dx + dy * dy);
   glTexCoord1f(r * scale);
}


static void
Draw(void)
{
   int i;

   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

   if (GradientMode == LINEAR) {
      GLfloat plane[4];
      plane[0] = 2.0 / WinWidth;
      plane[1] = 0.0;
      plane[2] = 0;
      plane[3] = 0;
      glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR);
      glTexGenfv(GL_S, GL_EYE_PLANE, plane);
      glEnable(GL_TEXTURE_GEN_S);
   }
   else {
      glDisable(GL_TEXTURE_GEN_S);
   }

   if (WrapMode == CLAMP)
      glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
   else
      glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_REPEAT);

   glEnable(GL_TEXTURE_1D);

   glPushMatrix();
   glTranslatef(Xpos, Ypos, 0);

   /* draw star */
   glBegin(GL_POLYGON);
   MyTexgen(0, 0);
   glVertex2f(0, 0);
   for (i = 0; i <= 10; i++) {
      float radius = 80.0 + 120.0 * (i & 1);
      float angle = i / 10.0 * PI * 2.0;
      float x = radius * sin(angle);
      float y = -radius * cos(angle);
      MyTexgen(x,y);
      glVertex2f(x, y);
   }
   glEnd();

   glPopMatrix();

   glDisable(GL_TEXTURE_1D);

   {
      static const char *gradients[] = { "Linear", "Radial" }; 
      static const char *wraps[] = { "Clamp", "Repeat" }; 
      char s[1000];
      sprintf(s, "Gradient Mode [g]: %s", gradients[GradientMode]);
      glWindowPos2iARB(10, 10);
      PrintString(s);

      sprintf(s, "Texture Wrap Mode [w]: %s", wraps[WrapMode]);
      glWindowPos2iARB(10, 30);
      PrintString(s);
   }

   glutSwapBuffers();
}


static void
Reshape(int width, int height)
{
   WinWidth = width;
   WinHeight = height;
   glViewport(0, 0, width, height);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glOrtho(0, width, 0, height, -1, 1);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
}


static void
Key(unsigned char key, int x, int y)
{
   (void) x;
   (void) y;
   switch (key) {
      case 'g':
         GradientMode = (GradientMode + 1) % 2;
         break;
      case 'w':
         WrapMode = (WrapMode + 1) % 2;
         break;
      case 27:
         exit(0);
         break;
   }
   glutPostRedisplay();
}


static GLboolean moving = GL_FALSE;
static int startX, startY;
static GLfloat startXpos, startYpos;


static void
Motion(int x, int y)
{
   if (moving) {
      float dx = x - startX;
      float dy = (WinHeight - y) - startY;
      Xpos = startXpos + dx;
      Ypos = startYpos + dy;
      glutPostRedisplay();
   }

}


static void
Mouse(int button, int state, int x, int y)
{
   if (button == GLUT_LEFT_BUTTON && state == GLUT_DOWN) {
      moving = GL_TRUE;
      startX = x;
      startY = WinHeight - y;
      startXpos = Xpos;
      startYpos = Ypos;
   }
   if (button == GLUT_LEFT_BUTTON && state == GLUT_UP) {
      moving = GL_FALSE;
   }
   glutPostRedisplay();
}



static void
Init(void)
{
   GLfloat gradient[256][3];
   GLint i;

   for (i = 0; i < 256; i++) {
      gradient[i][0] = 1.0 - i / 255.0;
      gradient[i][1] = i / 255.0;
      gradient[i][2] = 0.0;
   }
   glTexImage1D(GL_TEXTURE_1D, 0, GL_RGB, 256, 0, GL_RGB, GL_FLOAT, gradient);
   glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

   Xpos = 0.25 * WinWidth;
   Ypos = 0.25 * WinHeight;
}


int
main(int argc, char *argv[])
{
   glutInit(&argc, argv);
   glutInitWindowPosition(0, 0);
   glutInitWindowSize(WinWidth, WinHeight);
   glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH);
   glutCreateWindow(argv[0]);
   glutReshapeFunc(Reshape);
   glutKeyboardFunc(Key);
   glutMouseFunc(Mouse);
   glutMotionFunc(Motion);
   glutDisplayFunc(Draw);
   Init();
   glutMainLoop();
   return 0;
}
