
#include <assert.h>
#include <dlfcn.h>
#include <stdio.h>

#include "gl.h"
#include "libgl_internals.h"
#include "driver.h"


/* modify_ldt system call number:
 */
#define SYS_modify_ldt          123

/* modify_ldt system call params struct:
 */
struct modify_ldt_ldt_s {
    unsigned int entry_number;
    unsigned long base_addr;
    unsigned int limit;
    unsigned int seg_32bit:1;
    unsigned int contents:2;
    unsigned int read_exec_only:1;
    unsigned int limit_in_pages:1;
    unsigned int seg_not_present:1;
    unsigned int useable:1;
};

static __inline__ int modify_ldt(int func, void *ptr, unsigned long bytecount)
{
    int ret;

    __asm__ __volatile__("int $0x80"
                         : "=a" (ret)
                         : "0" (SYS_modify_ldt),
                           "b" (func),
                           "c" (ptr),
                           "d" (bytecount));

    return (ret >= 0) ? ret : -1;
}


#define INIT_THREAD_SELF(descr, nr)                                     \
{                                                                       \
    struct modify_ldt_ldt_s ldt_info = {                                \
        nr, (unsigned long)(descr), sizeof(*(descr)), 1, 0, 0, 0, 0, 1  \
    };                                                                  \
    int ret = modify_ldt(0x11, &ldt_info, sizeof(ldt_info));            \
    assert(ret >= 0);                                                   \
    __asm__ __volatile__ ("movl %0, %%gs" :: "q" ((nr << 3) | 0x7));    \
}

#define FREE_THREAD(descr, nr)                                          \
{                                                                       \
    struct modify_ldt_ldt_s ldt_info = {                                \
        nr, 0, 0, 0, 0, 1, 0, 1, 0                                      \
    };                                                                  \
    modify_ldt(0x11, &ldt_info, sizeof(ldt_info));                      \
}


/* A NOP dispatch layer, used whenever there is no context bound.
 */
#if 0
#define TRACE(func) printf("__nop_" #func "\n")
#else
#define TRACE(func)
#endif

static void __nop_Begin(GLenum mode)
{
    TRACE(Begin);
}
static void __nop_End(void)
{
    TRACE(End);
}
static void __nop_Normal3f(GLfloat x, GLfloat y, GLfloat z)
{
    TRACE(Normal3f);
}
static void __nop_TexCoord2f(GLfloat s, GLfloat t)
{
    TRACE(TexCoord2f);
}
static void __nop_Vertex3f(GLfloat x, GLfloat y, GLfloat z)
{
    TRACE(Vertex3f);
}

static __GLdispatchState __libgl_nop_dispatch = {
    __nop_Begin,
    __nop_End,
    __nop_Normal3f,
    __nop_TexCoord2f,
    __nop_Vertex3f
};


/* Our fake descriptor.  Usually, we would just access the regular
 * pthreads one.
 */
static struct pthread_descr __libgl_fake_descr;


/* Driver backend hooks.
 */
static void *__driver;
static void (*__driver_init_hook)(void);


static void __attribute__ ((constructor))
__libgl_init(void)
{
    struct pthread_descr *self = &__libgl_fake_descr;

    /* Set up the thread descriptor */
    self->p_header.data.self = self;

    INIT_THREAD_SELF(self, 1);

    /* Plug in the NOP dispatch */
    __libgl_set_current_dispatch(&__libgl_nop_dispatch);

    /* Open the driver backend */
    __driver = dlopen("driver.so", RTLD_LAZY);
    if (__driver == NULL) {
        printf("cannot open 'driver.so': %s\n", dlerror());
        exit(1);
    }

    /* Grab the driver init function */
    __driver_init_hook = (void (*)(void)) dlsym(__driver, "__driver_init");
    if (__driver_init_hook == NULL) {
        printf("cannot get symbol '__driver_init': %s\n", dlerror());
        exit(1);
    }

    /* Initialize the driver */
    __driver_init_hook();
}

static void __attribute__ ((destructor))
__libgl_fini(void)
{
    THREAD_SETUP(self);
    FREE_THREAD(self, 1);

    /* Close the driver backend */
    dlclose(__driver);
}
