
#include <assert.h>
#include <stdio.h>

#include "gl.h"
#include "libgl_internals.h"
#include "driver.h"


/* These would normally be hung off the context, but for the purpose
 * of this example we can just make them global variables.  Online
 * generated code would access memory in the same way as our hand-coded
 * assembly accesses these variables.
 *
 * Scratch layout:
 *
 *     scratch[0] = normal.x
 *     scratch[1] = normal.y
 *     scratch[2] = normal.z
 *     scratch[3] = texture[0].x
 *     scratch[4] = texture[0].y
 *
 * Buffer layout:
 *
 *     buffer[0] = position.x
 *     buffer[1] = position.y
 *     buffer[2] = position.z
 *     buffer[3] = normal.x
 *     buffer[4] = normal.y
 *     buffer[5] = normal.x
 *     buffer[6] = texture[0].x
 *     buffer[7] = texture[0].y
 */
GLfloat __driver_scratch[8];
GLfloat __driver_buffer[256 * 8];       /* Our vertex contains 8 floats */


void __glSetError(GLenum code)
{
    /* Do nothing here... */
}


/* Our driver's implementation of the API entrypoints.
 */

#if 0
#define TRACE(func) printf("__driver_" #func "\n")
#else
#define TRACE(func)
#endif

/* Optimally, you'd have two different dispatch tables -- one for
 * inside Begin/End, one for outside Begin/End.  That way, the error
 * checking is trivial.
 */
static void __driver_Begin(GLenum mode)
{
    __GL_SETUP();

    /* No validation needed, we would only be on this path if all state
     * changes have been dealt with.
     */

    if ((GLuint)mode > GL_POLYGON) {
        __glSetError(GL_INVALID_ENUM);
        return;
    }
}

static void __driver_End(void)
{
    TRACE(End);
}


/* Examples of runtime-generated API entrypoints.
 */
extern void __driver_x86_Normal3f(GLfloat x, GLfloat y, GLfloat z);
extern void __driver_x86_TexCoord2f(GLfloat s, GLfloat t);
extern void __driver_x86_Vertex3f(GLfloat x, GLfloat y, GLfloat z);

extern void __driver_sse2_Normal3f(GLfloat x, GLfloat y, GLfloat z);
extern void __driver_sse2_TexCoord2f(GLfloat s, GLfloat t);
extern void __driver_sse2_Vertex3f(GLfloat x, GLfloat y, GLfloat z);


/* Plain old boring C-code versions of the API entrypoints.
 */
static void __driver_Normal3f(GLfloat x, GLfloat y, GLfloat z)
{
    __GL_SETUP();
    TRACE(Normal3f);

    gc->state.current.normal.x = x;
    gc->state.current.normal.y = y;
    gc->state.current.normal.z = z;
}

static void __driver_TexCoord2f(GLfloat s, GLfloat t)
{
    __GL_SETUP();
    TRACE(TexCoord2f);

    gc->state.current.texture[0].x = s;
    gc->state.current.texture[0].y = t;
    gc->state.current.texture[0].z = 0.0f;
    gc->state.current.texture[0].w = 1.0f;
}

static void __driver_Vertex3f(GLfloat x, GLfloat y, GLfloat z)
{
    TRACE(Vertex3f);
}

static __GLdispatchState __driver_dispatch = {
    __driver_Begin,
    __driver_End,
    __driver_Normal3f,
    __driver_TexCoord2f,
    __driver_Vertex3f
};

static __GLcontext __driver_context;


void __driver_flush(void)
{
    THREAD_SETUP(self);

    /* Usually need to do something with the buffered data...
     */
    THREAD_SETMEM(self, p_libgl_specific[2], __driver_buffer);
    THREAD_SETMEM(self, p_libgl_specific[3], (void *) 256);
}

void __driver_init(void)
{
    printf("initializing driver backend... ");

    memset(&__driver_context, 0, sizeof(__driver_context));

    /* Conditionally plug in the asm entrypoint backends.
     */
    if (getenv("USE_X86_CODE")) {
        __driver_dispatch.Normal3f = __driver_x86_Normal3f;
        __driver_dispatch.TexCoord2f = __driver_x86_TexCoord2f;
        __driver_dispatch.Vertex3f = __driver_x86_Vertex3f;
    }

    if (getenv("USE_SSE2_CODE")) {
        __driver_dispatch.Normal3f = __driver_sse2_Normal3f;
        __driver_dispatch.TexCoord2f = __driver_sse2_TexCoord2f;
        __driver_dispatch.Vertex3f = __driver_sse2_Vertex3f;
    }

    if (__driver_dispatch.Vertex3f == __driver_x86_Vertex3f) {
        printf("using x86 api... ");
    } else if (__driver_dispatch.Vertex3f == __driver_sse2_Vertex3f) {
        printf("using sse2 api... ");
    }

    /* To make this example simple, we only have a single context and
     * its corresponding dispatch table.
     */
    __libgl_set_current_context(&__driver_context);
    __libgl_set_current_dispatch(&__driver_dispatch);

    __driver_flush();

    printf("done.\n");
}
